from collections import defaultdict
import logging

from django.db import transaction
from django.utils.functional import cached_property

from plan.suspicion.models import (
    Issue,
    IssueGroupThreshold,
    ServiceIssue,
    ServiceTrafficStatus,
)


log = logging.getLogger(__name__)


def create_new_service_issue_groups():
    existed_service_issue_groups = set(
        ServiceIssue.objects
        .active()
        .filter(issue_group__isnull=False)
        .values_list('issue_group', 'service_id')
    )
    necessary_service_issue_groups = set(
        ServiceIssue.objects
        .problem()
        .filter(issue__isnull=False)
        .values_list('issue__issue_group_id', 'service_id')
    )
    service_issue_groups_to_create = [
        ServiceIssue(state=ServiceIssue.STATES.ACTIVE, issue_group_id=issue_group_id, service_id=service_id)
        for issue_group_id, service_id
        in necessary_service_issue_groups - existed_service_issue_groups
    ]
    if service_issue_groups_to_create:
        ServiceIssue.objects.bulk_create(service_issue_groups_to_create)


def get_max_issue_groups_weights():
    issue_group_to_max_weight = defaultdict(float)
    for issue in Issue.objects.active():
        issue_group_to_max_weight[issue.issue_group_id] += issue.weight

    return issue_group_to_max_weight


class IssueGroupsChecker(object):
    BATCH_SIZE = 500

    def __init__(self):
        create_new_service_issue_groups()
        self.service_ids = set()
        self.issue_groups_ids = set()
        self.service_and_issue_group = []
        self.problem_issues_by_service_and_groups = defaultdict(dict)
        self.fill_problem_issues()
        self.traffic_by_services_and_groups = defaultdict(dict)
        self.fill_traffic_status_by_services_and_groups()
        self.issue_group_to_max_weight = get_max_issue_groups_weights()
        self.issue_group_levels = defaultdict(list)
        self.fill_issue_group_levels()

    def fill_issue_group_levels(self):
        # ToDo: добавить фильтры, если IssueGroupThreshold станет много
        issue_groups_thresholds = (
            IssueGroupThreshold.objects
            .order_by('threshold')
            .values_list('issue_group_id', 'threshold')
        )
        for issue_group, threshold in issue_groups_thresholds:
            self.issue_group_levels[issue_group].append(threshold)

    @cached_property
    def issues_by_services_and_groups(self):
        issues_by_services = defaultdict(lambda: defaultdict(list))
        problem_issues = (
            ServiceIssue.objects
            .problem()
            .filter(issue__issue_group__isnull=False)
            .select_related('issue')
        )
        for service_issue in problem_issues:
            issues_by_services[service_issue.service_id][service_issue.issue.issue_group_id].append(service_issue)
        return issues_by_services

    def fill_problem_issues(self):
        problem_services_issues_groups = (
            ServiceIssue.objects
            .problem()
            .for_alive_services()
            .filter(issue_group__isnull=False)
            .select_related('issue_group')
        )

        for issues_group in problem_services_issues_groups:
            self.service_ids.add(issues_group.service_id)
            self.issue_groups_ids.add(issues_group.issue_group_id)
            self.service_and_issue_group.append((issues_group.service_id, issues_group.issue_group_id))
            self.problem_issues_by_service_and_groups[issues_group.service_id][issues_group.issue_group_id] = issues_group

    def fill_traffic_status_by_services_and_groups(self):
        traffic_statuses = ServiceTrafficStatus.objects.filter(
            service_id__in=self.service_ids,
            issue_group_id__in=self.issue_groups_ids,
        )
        for status in traffic_statuses:
            self.traffic_by_services_and_groups[status.service_id][status.issue_group_id] = status

    def check_one_service_traffic(self, issue, service_issues, max_weight, service_traffic_status, thresholds):
        # ToDo: создавать service_traffic_status батчем
        if service_traffic_status is None:
            service_traffic_status = ServiceTrafficStatus.objects.create(
                service_id=issue.service_id,
                issue_group=issue.issue_group,
            )
        weight, issue_count = issue.get_weight_and_count(service_issues_for_group=service_issues, max_weight=max_weight)
        new_level = service_traffic_status.get_new_level(value=weight, issue_group_thresholds=thresholds)
        if service_traffic_status.level != new_level:
            ServiceIssue.objects.filter(service_id=issue.service_id, issue_group=issue.issue_group).set_not_processed()
        service_traffic_status.update_fields(level=new_level, current_weight=weight, issue_count=issue_count)

    def process_one_batch(self, batch):
        for service, issues_group in batch:
            service_issue = self.problem_issues_by_service_and_groups[service][issues_group]
            try:
                traffic = self.traffic_by_services_and_groups[service][issues_group]
            except (KeyError, TypeError):
                traffic = None
            try:
                self.check_one_service_traffic(
                    issue=service_issue,
                    service_issues=self.issues_by_services_and_groups[service][issues_group],
                    max_weight=self.issue_group_to_max_weight[issues_group],
                    service_traffic_status=traffic,
                    thresholds=self.issue_group_levels[issues_group],
                )
            except Exception:
                log.error('Error while check servise_issue %s', service_issue)

    def run(self):
        for batch_index in range(0, len(self.service_and_issue_group), self.BATCH_SIZE):
            with transaction.atomic():
                self.process_one_batch(self.service_and_issue_group[batch_index:batch_index + self.BATCH_SIZE])
