import logging
import os
import yenv

from drf_yasg import openapi
from drf_yasg.generators import OpenAPISchemaGenerator
from drf_yasg.inspectors.view import SwaggerAutoSchema
from drf_yasg.views import get_schema_view

from functools import lru_cache
from library.python import resource
from rest_framework import permissions


log = logging.getLogger(__name__)


class SchemaGenerator(OpenAPISchemaGenerator):
    user_is_superuser = False

    def get_overrides(self, view, method):
        overrides = super(SchemaGenerator, self).get_overrides(view, method)
        has_auto_schema = overrides.get('auto_schema', False)

        # т.е. в автосхеме лежит False => в overrides ничего нет про схему
        if not has_auto_schema and has_auto_schema is not None:
            overrides['auto_schema'] = self.get_swagger_schema(view)

        return overrides

    def get_swagger_schema(self, view):
        default_swagger_schema = getattr(view, 'default_swagger_schema', None)

        if default_swagger_schema == SwaggerFrontend and not self.user_is_superuser:
            return None

        return default_swagger_schema

    def get_endpoints(self, request):
        endpoints = super(SchemaGenerator, self).get_endpoints(request)

        user = request.person.user
        self.user_is_superuser = user.is_superuser

        return endpoints


class BaseSwaggerSchema(SwaggerAutoSchema):
    """
    Базовый класс для разделов документации.
    Отнаследованные классы должны быть прописаны у view в default_swagger_schema.
    По умолчанию все вьюхи не попадают в документацию.
    Чтобы отключить не всю вьюху, а только конкретную ручку, нужно использовать для нее
    декоратор @swagger_auto_schema(auto_schema=None)
    Аналогичным способом можно включить во вьюхе только одну ручку.
    """
    new_tags = None

    def __init__(self, *args, **kwargs):
        super(BaseSwaggerSchema, self).__init__(*args, **kwargs)
        if not self.new_tags:
            return

        tags = self.overrides.get('tags')
        if not tags:
            self.overrides['tags'] = self.new_tags
        else:
            self.overrides['tags'].extend(self.new_tags)

    def get_response_schemas(self, *args, **kwargs):
        responses = super(BaseSwaggerSchema, self).get_response_schemas(*args, **kwargs)
        view = self.view
        default_fields = getattr(view, 'default_fields', None)
        method = self.method.lower()

        if default_fields and method == 'get':
            for cod, Response in responses.items():
                Response.description = f'По умолчанию в ответе выдаются только поля: {default_fields}. \n' \
                                       f'Используйте параметр ``fields`` - список полей, которые надо вернуть в ответе, ' \
                                       f'через запятую.'

        return responses


class SwaggerFrontend(BaseSwaggerSchema):
    """
    API для фронта
    """
    new_tags = ['frontend']


class SwaggerDuty(BaseSwaggerSchema):
    """
    API Дежурств
    """
    new_tags = ['duty']


class SwaggerServices(BaseSwaggerSchema):
    """
    Сервисное API
    """
    new_tags = ['services']


class SwaggerResources(BaseSwaggerSchema):
    """
    API ресурсов
    """
    new_tags = ['resources']


def extract_resources(prefix):
    for res, data in resource.iteritems(prefix='resfs/file/{}'.format(prefix), strip_prefix=True):
        path = os.path.join(prefix, res[1:])
        if not os.path.exists(path):
            os.makedirs(prefix)
            with open(path, 'wb') as f:
                f.write(data)


@lru_cache()
def get_description():
    if yenv.type == 'development':
        return ''
    extract_resources('intranet/plan/src/plan/swagger')
    path = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'swagger/swagger_description.md')
    with open(path) as file:
        description = file.read()
    return description


schema_view = get_schema_view(
    openapi.Info(
        title="ABC API",
        default_version='v4',
        description=get_description(),
    ),
    public=True,
    generator_class=SchemaGenerator,
    permission_classes=(permissions.IsAuthenticated,),
)
