import logging
from django.conf import settings
from django.db.models import (
    Count, F, Q, IntegerField,
    Subquery, OuterRef,
)
from django.utils import timezone

from plan.common.utils.tasks import lock_task
from plan.metrics.staff import get_found_in_staff_times
from plan.resources.constants import FOUND_IN_STAFF_TIME_LIMITS
from plan.resources.models import ServiceResource, Resource
from plan.oebs.models import OEBSAgreement, STATES
from plan.oebs.constants import OEBS_DEVIATIONS_REASONS
from plan.services.models import (
    Service,
    ServiceMember,
    ServiceMoveRequest,
    ServiceDeleteRequest,
    ServiceCloseRequest,
    ServiceRequestMixin,
)
from plan.staff.models import Department
from plan.unistat.models import MetricsRecord, TaskMetric


log = logging.getLogger(__name__)


def closuretree_damaged_tree():
    problematic_slugs = list(
        Service._closure_model.objects
        .filter(depth=1)
        .annotate(object_parent_id=F('child__parent_id'))
        .exclude(parent_id=F('object_parent_id'))
        .values_list('child__slug', flat=True)
    )
    if problematic_slugs:
        log.error('Found closuretree damages for %s', ', '.join(problematic_slugs))
    return len(problematic_slugs)


def service_level_broken():
    bad_services = Service.objects.filter(
        Q(parent=None, level__gt=0) |
        ~Q(level=F('parent__level') + 1)
    )
    issues = bad_services.count()
    if issues > 0:
        slugs = list(bad_services.values_list('slug', flat=True))
        log.error('Found broken level at services %s', ', '.join(slugs))
        fix_service_closuretree_levels.apply_async(args=slugs)
    return issues


@lock_task
def fix_service_closuretree_levels(*service_slugs):
    services = Service.objects.filter(slug__in=service_slugs)
    _fix_closuretree_levels(services)


@lock_task
def fix_broken_departments(*department_staff_id_list):
    departments = Department.objects.filter(staff_id__in=department_staff_id_list)
    _fix_closuretree_levels(departments)


def _fix_closuretree_levels(queryset):
    for node in queryset:
        subtree_with_self = node.get_descendants(include_self=True).order_by('level')
        subtree_without_self = node.get_descendants().order_by('level')

        # проапдейтить level
        old_level = node.level
        new_level = 0 if node.is_root_node() else node.parent.level + 1
        leveldiff = new_level - old_level
        subtree_with_self.update(level=F('level') + leveldiff)

        # проапдейтить ссылки
        links = node._closure_model.objects.filter(child_id__in=subtree_with_self)
        links.delete()
        cached_subtree = [node] + list(subtree_without_self)
        for item in cached_subtree:
            item._closure_createlink()


def broken_department():
    departments = Department.objects.filter(intranet_status=1).order_by('level')
    broken = []
    for department in departments:
        if not department.is_root_node() and len(department.get_ancestors()) == 0:
            broken.append(department.staff_id)
    if broken:
        log.error('Found broken departments: %s', ', '.join(str(s) for s in broken))
        fix_broken_departments.apply_async(args=broken)
    return len(broken)


def get_oebs_deviations() -> dict:
    deviations = {
        'oebs_deviation': 0,
    }
    for key in [
        OEBS_DEVIATIONS_REASONS.FLAG,
        OEBS_DEVIATIONS_REASONS.RESOURCE,
        OEBS_DEVIATIONS_REASONS.PARENT,
        OEBS_DEVIATIONS_REASONS.NAME,
    ]:
        deviations[f'oebs_deviation_{key}'] = 0

    services_with_deviations = Service.objects.oebs_deviation()
    for service in services_with_deviations:
        deviations['oebs_deviation']+=1
        deviations[f'oebs_deviation_{service.oebs_data["deviation_reason"]}']+=1
    return deviations


def get_tvm2_granted_duplicates():
    # Количество выданных ServiceResource в статусе granted
    # для каждого ресурса, не должно превышать одного
    return Resource.objects.filter(
        type__code=settings.TVM_RESOURCE_TYPE_CODE
    ).annotate(
        granted=Subquery(
            ServiceResource.objects.filter(
                state=ServiceResource.GRANTED,
                resource=OuterRef('pk')
            ).values('resource')
                .annotate(cnt=Count('pk'))
                .values('cnt'),
            output_field=IntegerField()
        )
    ).filter(granted__gt=1).count()


def time_at_staff_states():
    percentiles = [90, 95, 100]
    result = {}
    for limit_name, limit_timedelta in FOUND_IN_STAFF_TIME_LIMITS.items():
        found_in_staff_time_delta = sorted(get_found_in_staff_times(limit_timedelta))
        qs_size = len(found_in_staff_time_delta)
        for percentile in percentiles:
            index = int(qs_size * percentile // 100) - 1
            name = 'found_in_staff_time_for_{}_hours_{}_percentile'.format(limit_name, str(percentile))
            result[name] = found_in_staff_time_delta[index].total_seconds() if found_in_staff_time_delta else 0
    return result


def calculate_granted_resources():
    granted_resources = ServiceResource.objects.filter(
        state='granted',
        resource__type__code__isnull=False
    ).values('resource__type__code').annotate(count=Count('resource_id'))
    return {
        'granted_resources_{}'.format(item['resource__type__code']): item['count']
        for item in granted_resources
    }


def calculate_unistat_metrics():
    meta_other = Service.objects.get(slug=settings.ABC_DEFAULT_SERVICE_PARENT_SLUG)
    active_services = Service.objects.filter(state__in=Service.states.ACTIVE_STATES)

    active_with_roles = set(active_services.filter(idm_roles_count__gt=0).values_list('id', flat=True))
    active_with_puncher_rules = set(active_services.filter(puncher_rules_count__gt=0).values_list('id', flat=True))
    active_with_active_resources = set(ServiceResource.objects.granted().values_list('service__id', flat=True).distinct())

    base_service_under_usual = Service.objects.alive().base().filter(parent__is_base=False).count()
    base_non_leaf = Service.objects.base_non_leaf()
    usual_service_under_base_non_leaf = Service.objects.active().filter(is_base=False, parent__in=base_non_leaf).count()

    dispenser_changing_ids = set()

    for model in [ServiceCloseRequest, ServiceDeleteRequest]:
        dispenser_changing_ids.update(
            model.objects.filter(
                state=ServiceRequestMixin.PROCESSING_D,
                service__state__in=Service.states.ACTIVE_STATES,
            ).values_list('service_id', flat=True)
        )

    services_in_readonly_state_for_long_time = active_services.filter(
        ~Q(id__in=dispenser_changing_ids) &
        Q(
            readonly_state__isnull=False,
            readonly_start_time__lt=timezone.now() - timezone.timedelta(
                minutes=settings.MINUTES_BEFORE_SERVICE_DIE_IN_READ_ONLY_STATE
            ),
        )
    )

    # основные метрики
    metrics = {
        'total_services': active_services.count(),
        'services_with_roles': len(active_with_roles),
        'services_with_membership_inheritance': Service.objects.filter(membership_inheritance=True).count(),
        'services_with_puncher_rules': len(active_with_puncher_rules),
        'services_with_active_resources': len(active_with_active_resources),
        'services_with_something_active': len(active_with_roles | active_with_puncher_rules | active_with_active_resources),
        'meta_other_services': (
            meta_other.get_descendants()
            .filter(state__in=Service.states.ACTIVE_STATES)
            .count()
        ),
        'non_exportable_services': active_services.filter(is_exportable=False).count(),
        'ownerless_services': active_services.filter(owner=None).count(),
        'live_move_requests': (
            ServiceMoveRequest.objects
            .exclude(state__in=ServiceMoveRequest.INACTIVE_STATES)
            .count()
        ),
        'active_memberships': ServiceMember.objects.team().count(),
        'services_in_readonly_state_for_long_time': services_in_readonly_state_for_long_time.count(),
        'unique_service_members': (
            ServiceMember.objects
            .values_list('staff__id', flat=True)
            .distinct().count()
        ),
        'robot_memberships': ServiceMember.objects.filter(role__code='robot').count(),
        'unique_robots': (
            ServiceMember.objects
            .filter(role__code='robot')
            .values_list('staff__id', flat=True)
            .distinct().count()
        ),
        'depriving_deprived_members': ServiceMember.objects.filter(
            state=ServiceMember.states.DEPRIVING,
            deprived_at__isnull=False,
        ).count(),
        'services_with_important_resources': (
            ServiceResource.objects
            .filter(type__is_important=True)
            .granted()
            .values_list('service__id', flat=True)
            .distinct().count()
        ),
        'center_last_touched': 0,

        'base_service_under_usual': base_service_under_usual,
        'usual_service_under_base_non_leaf': usual_service_under_base_non_leaf,
        'closuretree_damaged_tree': closuretree_damaged_tree(),
        'service_level_broken': service_level_broken(),
        'broken_department': broken_department(),
        'oebs_stale_agreements': OEBSAgreement.objects.filter(
            state=STATES.APPLIED_IN_OEBS,
            updated_at__lt=timezone.now() - timezone.timedelta(minutes=20)
        ).count(),

    }
    metrics.update(get_oebs_deviations())
    meta_other_services_with_sandbox = meta_other.get_descendants(include_self=True).active()
    non_exportable_in_sandbox = meta_other_services_with_sandbox.filter(is_exportable=False).count()
    metrics['exportable_in_sandbox'] = meta_other_services_with_sandbox.count() - non_exportable_in_sandbox
    metrics['non_exportable_services_not_in_sandbox'] = metrics['non_exportable_services'] - non_exportable_in_sandbox
    metrics['tvm2_granted_duplicates'] = get_tvm2_granted_duplicates()
    metrics.update(time_at_staff_states())
    metrics.update(calculate_granted_resources())

    # сколько секунд прошло с последнего успешного выполнения тасок
    for task_metric in TaskMetric.objects.filter(send_to_unistat=True):
        metrics[task_metric.task_name] = (timezone.now() - task_metric.last_success_end).total_seconds()

    # считаем сколько сервисов в каждом статусе
    services_by_state = (
        (
            'readonly_state',
            [state for state, _ in Service.READONLY_STATES],
            active_services
            .exclude(readonly_state__isnull=True)
            .values('readonly_state')
            .annotate(Count('id'))
        ),
    )

    for prefix, states, queryset in services_by_state:
        # добавляем изначальные значения, чтоб в выдаче всегда был хотя бы 0
        for state in states:
            metrics['services_in_{}_{}'.format(prefix, state)] = 0

        for data in queryset:
            state = data[prefix]
            count = data['id__count']

            metrics['services_in_{}_{}'.format(prefix, state)] = count

    return metrics


@lock_task(bind=True)
def populate_unistat_metrics(self):
    # done
    metrics_record, created = MetricsRecord.objects.get_or_create()
    metrics_record.metrics.update(calculate_unistat_metrics())
    metrics_record.save()
