from functools import wraps

import django
from django_pgaas.wsgi import IdempotentWSGIHandler
from django_tools_log_context import request_log_context

from plan.entrypoints import setup_env

setup_env()


class LogContextWSGIHandler(IdempotentWSGIHandler):
    """Custom WSGIHangler to wrap views with request_log_context"""
    def make_view_atomic(self, view):
        log_context = True
        view_cls = getattr(view, 'cls', None)
        if view_cls:
            log_context = getattr(view_cls, 'log_context', True)
        view = super(LogContextWSGIHandler, self).make_view_atomic(view)

        if log_context:
            @wraps(view)
            def wrapped(request, *args, **kwargs):
                with request_log_context(request, endpoint=view) as log_context:
                    response = view(request, *args, **kwargs)
                    log_context.context['request']['status_code'] = response.status_code
                    return response
            return wrapped
        else:
            return view


def get_wsgi_application():
    django.setup()
    return LogContextWSGIHandler()


application = get_wsgi_application()
