# coding: utf-8

import time
from datetime import datetime
from xml.etree.ElementTree import XMLParser

import cx_Oracle
import psycopg2
import schedule
from psycopg2.extras import execute_values

from . import settings

conn_oracle = None
conn_pg = None


def init_db():
    global conn_oracle, conn_pg

    if conn_oracle is None:
        conn_oracle = cx_Oracle.connect(
            settings.ORACLE_USER, settings.ORACLE_PASS, settings.ORACLE_TNS
        )

    if conn_pg is None:
        conn_pg = psycopg2.connect(
            host=settings.PG_HOST,
            dbname=settings.PG_DBNAME,
            user=settings.PG_USER,
            password=settings.PG_PASS,
            port=settings.PG_PORT,
            target_session_attrs='read-write',
        )


def sync_generic(oracle_table, procu_table):
    init_db()

    oracle = conn_oracle.cursor()
    data_from = dict(
        oracle.execute(
            f'SELECT id, name FROM {oracle_table} WHERE name IS NOT NULL'
        )
    )
    oracle.close()

    with conn_pg, conn_pg.cursor() as cursor:
        cursor.execute(
            f"SELECT key, name FROM {procu_table} WHERE is_deleted='f';"
        )
        data_to = dict(cursor.fetchall())

    # Delete
    with conn_pg, conn_pg.cursor() as cursor:
        for key in data_to.keys() - data_from.keys():
            cursor.execute(
                f"UPDATE {procu_table} " "SET is_deleted='t' WHERE key=%s;",
                (key,),
            )

    # Update
    with conn_pg, conn_pg.cursor() as cursor:
        for key in data_to.keys() & data_from.keys():
            if data_to[key] == data_from[key]:
                continue

            cursor.execute(
                f"UPDATE {procu_table} " "SET name=%s WHERE key=%s;",
                (data_from[key], key),
            )

    # Create
    values = [
        (key, data_from[key], False)
        for key in data_from.keys() - data_to.keys()
        if data_from[key]
    ]

    with conn_pg, conn_pg.cursor() as cursor:
        execute_values(
            cursor,
            f"INSERT INTO {procu_table} (key, name, is_deleted) VALUES %s "
            "ON CONFLICT (key) DO UPDATE SET is_deleted='f', name=EXCLUDED.name",
            values,
        )


def sync_cfo():
    sync_generic('T_CFO', 'api_oraclecfo')


def sync_program():
    sync_generic('T_PROGRAM', 'api_oracleprogram')


def sync_project():
    sync_generic('T_PROJECTS', 'api_oracleproject')


def sync_task():
    sync_generic('T_TASKS', 'api_oracletask')


def sync_mvp():
    sync_generic('T_MVP', 'api_oraclemvp')


def sync_service():
    sync_generic('T_GL_SERVICE', 'api_oracleservice')


def sync_purchase_group():
    sync_generic('T_PURCHASE_GROUP', 'api_oraclepurchasegroup')


def sync_budget_line():
    sync_generic('T_BUDGET_LINE', 'api_oraclebudgetline')


def sync_system():
    sync_generic('T_SYSTEM', 'api_oraclesystem')


def sync_subsystem():
    sync_generic('T_SUBSYSTEM', 'api_oraclesubsystem')


def sync_company():
    init_db()

    procu_table = 'api_oraclecompany'
    oracle_table = 'T_COMPANY'

    # ---------------------

    oracle = conn_oracle.cursor()
    cs = oracle.execute(
        f'SELECT orgid, inn, name, shortname  name FROM {oracle_table}'
    )
    data_from = {c[0]: c[1:] for c in cs}
    oracle.close()

    # ---------------------

    with conn_pg, conn_pg.cursor() as cursor:
        cursor.execute(
            f"SELECT orgid, inn, name, shortname FROM {procu_table} "
            f"WHERE is_deleted='f';"
        )
        data_to = {c[0]: c[1:] for c in cursor.fetchall()}

    # Delete
    with conn_pg, conn_pg.cursor() as cursor:
        for orgid in data_to.keys() - data_from.keys():
            cursor.execute(
                f"UPDATE {procu_table} " f"SET is_deleted='t' WHERE orgid=%s;",
                (orgid,),
            )

    # Update
    with conn_pg, conn_pg.cursor() as cursor:
        for orgid in data_to.keys() & data_from.keys():
            if data_to[orgid] == data_from[orgid]:
                continue

            cursor.execute(
                f"UPDATE {procu_table} "
                f"SET inn=%s, name=%s, shortname=%s WHERE key=%s;",
                (*data_from[orgid], orgid),
            )

    # Create
    values = [
        (orgid, *data_from[orgid], False)
        for orgid in data_from.keys() - data_to.keys()
        if data_from[orgid]
    ]

    with conn_pg, conn_pg.cursor() as cursor:
        execute_values(
            cursor,
            f"INSERT INTO {procu_table} "
            f"(orgid, inn, name, shortname, is_deleted) VALUES %s "
            f"ON CONFLICT (orgid) DO UPDATE SET is_deleted='f';",
            values,
        )


def sync_fk_program_project():
    init_db()

    with conn_pg, conn_pg.cursor() as cursor:
        cursor.execute(
            """
            SELECT key, id FROM api_oracleprogram WHERE is_deleted='f';
            """
        )
        programs_to = dict(cursor.fetchall())

        cursor.execute(
            """
            SELECT id, key, program_id FROM api_oracleproject
            WHERE is_deleted='f';
            """
        )
        projects_to = list(cursor.fetchall())

    oracle = conn_oracle.cursor()
    projects_from = dict(
        oracle.execute(
            """
            SELECT id, program_id FROM T_PROJECTS WHERE program_id IS NOT NULL
            """
        )
    )
    oracle.close()

    # Update
    with conn_pg, conn_pg.cursor() as cursor:
        for project_id, project_key, program_id_to in projects_to:

            try:
                program_key = projects_from[project_key]
                program_id_from = programs_to[program_key]
            except KeyError:
                continue

            if program_id_to == program_id_from:
                continue

            cursor.execute(
                "UPDATE api_oracleproject SET program_id=%s WHERE id=%s;",
                (program_id_from, project_id),
            )


def sync_fk_system_subsystem():
    init_db()

    with conn_pg, conn_pg.cursor() as cursor:
        cursor.execute(
            "SELECT key, id FROM api_oraclesystem WHERE is_deleted='f';"
        )
        systems_to = dict(cursor.fetchall())

        cursor.execute(
            "SELECT id, key, system_id FROM api_oraclesubsystem "
            "WHERE is_deleted='f';"
        )
        subsystems_to = list(cursor.fetchall())

    oracle = conn_oracle.cursor()
    subsystems_from = dict(
        oracle.execute(
            """
            SELECT id, system_id FROM T_SUBSYSTEM WHERE system_id IS NOT NULL
            """
        )
    )
    oracle.close()

    # Update
    with conn_pg, conn_pg.cursor() as cursor:
        for subsystem_id, subsystem_key, system_id_to in subsystems_to:

            try:
                system_key = subsystems_from[subsystem_key]
                system_id_from = systems_to[system_key]
            except KeyError:
                continue

            if system_id_to == system_id_from:
                continue

            cursor.execute(
                "UPDATE api_oraclesubsystem SET system_id=%s WHERE id=%s;",
                (system_id_from, subsystem_id),
            )


def sync_fk_project_task():
    init_db()

    with conn_pg, conn_pg.cursor() as cursor:
        cursor.execute(
            "SELECT key, id FROM api_oracleproject WHERE is_deleted='f';"
        )
        projects_to = dict(cursor.fetchall())

        cursor.execute(
            "SELECT id, key, project_id FROM api_oracletask "
            "WHERE is_deleted='f';"
        )
        tasks_to = list(cursor.fetchall())

    oracle = conn_oracle.cursor()
    tasks_from = dict(
        oracle.execute(
            'SELECT id, project_id FROM T_TASKS WHERE project_id IS NOT NULL'
        )
    )
    oracle.close()

    # Update
    with conn_pg, conn_pg.cursor() as cursor:
        for task_id, task_key, project_id_to in tasks_to:

            try:
                project_key = tasks_from[task_key]
                project_id_from = projects_to[project_key]
            except KeyError:
                continue

            if project_id_to == project_id_from:
                continue

            cursor.execute(
                "UPDATE api_oracletask SET project_id=%s WHERE id=%s;",
                (project_id_from, task_id),
            )


def sync_fk_company_project():
    init_db()

    with conn_pg, conn_pg.cursor() as cursor:
        cursor.execute(
            "SELECT orgid, id FROM api_oraclecompany WHERE is_deleted='f';"
        )
        companies_to = dict(cursor.fetchall())

        cursor.execute(
            "SELECT id, key, company_id FROM api_oracleproject "
            "WHERE is_deleted='f';"
        )
        projects_to = list(cursor.fetchall())

    oracle = conn_oracle.cursor()
    projects_from = dict(
        oracle.execute(
            'SELECT id, org_id FROM T_PROJECTS WHERE org_id IS NOT NULL'
        )
    )
    oracle.close()

    # Update
    with conn_pg, conn_pg.cursor() as cursor:
        for project_id, project_key, company_id_to in projects_to:

            try:
                company_key = projects_from[project_key]
                company_id_from = companies_to[int(company_key)]
            except KeyError:
                continue

            if company_id_to == int(company_id_from):
                continue

            cursor.execute(
                "UPDATE api_oracleproject SET company_id=%s WHERE id=%s;",
                (company_id_from, project_id),
            )


class SuppliersTarget:
    FIELD_MAP = {'ID': 0, 'Name': 1, 'FullName': 2, 'INN': 3, 'Deleted': 4}

    def __init__(self):
        self.objects = []
        self.current = None
        self.field = None
        self.value = None

    def start(self, tag, attrib):

        if tag == 'ListItem':
            self.field = field = self.FIELD_MAP.get(attrib['FieldName'])
            if field is not None:
                self.value = []

        elif tag == 'Element':
            self.current = [None, None, None, None, None]

    def end(self, tag):
        if tag == 'ListItem':
            if self.field is not None:
                self.current[self.field] = ''.join(self.value)

        elif tag == 'Element':
            self.objects.append(self.current)

    def data(self, data):
        if self.field is not None:
            self.value.append(data)

    def close(self):

        data = {}

        for obj in self.objects:
            obj[4] = obj[4] == '1'
            obj[3] = obj[3].strip()

            # Deduplicate rows by key
            data[obj[0]] = obj

        return list(data.values())


def sync_suppliers():
    init_db()
    import_log_key = 'suppliers'

    with conn_pg, conn_pg.cursor() as cursor:

        cursor.execute(
            "SELECT last_id FROM api_oracleimportlog " "WHERE key=%s",
            (import_log_key,),
        )

        try:
            (last_object_id,) = cursor.fetchone()

        except TypeError:
            last_object_id = 0

        oracle_cursor = conn_oracle.cursor()
        oracle_cursor = oracle_cursor.execute(
            "SELECT OBJECTID, TO_CHAR(XMLDOCUMENT) FROM T_XML_EXCHANGE_PROCU "
            "WHERE DESTLISTNAME='Справочник Контрагенты' AND OBJECTID>:lastid "
            "ORDER BY OBJECTID",
            lastid=last_object_id,
        )

        while True:
            rows = oracle_cursor.fetchmany(5000)

            if not rows:
                break

            parser = XMLParser(target=SuppliersTarget())
            parser.feed('<Suppliers>')

            for row in rows:
                last_object_id = row[0]
                parser.feed(row[1])

            parser.feed('</Suppliers>')

            values = parser.close()

            execute_values(
                cursor,
                "INSERT INTO api_oraclesupplier "
                "(key, name, full_name, inn, is_deleted) "
                "VALUES %s "
                "ON CONFLICT (key) "
                "DO UPDATE SET "
                "name=EXCLUDED.name, "
                "full_name=EXCLUDED.full_name, "
                "inn=EXCLUDED.inn, "
                "is_deleted=EXCLUDED.is_deleted"
                ";",
                values,
            )

            print('Supplier import in progress, last id: ', last_object_id)

        cursor.execute(
            "INSERT INTO api_oracleimportlog "
            "(key, last_id, last_updated_at) "
            "VALUES (%s, %s, %s) "
            "ON CONFLICT (key) "
            "DO UPDATE SET "
            "last_id=EXCLUDED.last_id, "
            "last_updated_at=EXCLUDED.last_updated_at;",
            (import_log_key, last_object_id, datetime.now()),
        )

    oracle_cursor.close()


def sync_all():
    print('Start sync...')
    sync_cfo()
    sync_program()
    sync_project()
    sync_task()
    sync_mvp()
    sync_service()
    sync_company()
    sync_purchase_group()
    sync_budget_line()
    sync_system()
    sync_subsystem()
    sync_fk_program_project()
    sync_fk_project_task()
    sync_fk_company_project()
    sync_fk_system_subsystem()
    sync_suppliers()
    print('Success!')


schedule.every(2).hours.do(sync_all)


def main():
    while True:
        schedule.run_pending()
        time.sleep(60)


if __name__ == '__main__':
    main()
