# coding: utf-8

import logging
from datetime import timedelta

from blackbox import Blackbox
from django.conf import settings
from django.contrib.auth import authenticate, login
from django.contrib.auth.backends import ModelBackend
from django.core.exceptions import PermissionDenied
from django.utils.timezone import now
from django_yauth.authentication_mechanisms import cookie, oauth, tvm

from procu.api import models
from procu.api.utils import is_internal

logger = logging.getLogger(__name__)


def _get_blackbox_fields(uid):
    blackbox = getattr(settings, 'YAUTH_BLACKBOX_INSTANCE') or Blackbox()
    fields = dict(settings.YAUTH_PASSPORT_FIELDS)
    try:
        info = blackbox.userinfo(
            uid_or_login=uid,
            userip='127.0.0.1',
            by_login=False,
            dbfields=fields,
            emails='getdefault',
        )
    except Exception:
        raise PermissionDenied('Could not fetch user from Blackbox')

    if not info['uid']:
        raise PermissionDenied(f'UID {uid} not found')

    return info['fields']


def get_username_by_uid(uid):
    fields = _get_blackbox_fields(uid)
    return fields['username']


class CookieAuthBackend(cookie.Mechanism, ModelBackend):

    mechanism_name = 'cookie'

    # noinspection PyMethodOverriding
    def authenticate(self, request):
        if not (settings.IS_DEVELOPMENT or is_internal(request)):
            return None

        return super().authenticate(request)


class CapabilityURLBackend(ModelBackend):

    # noinspection PyMethodOverriding
    def authenticate(self, request, token):
        try:
            later_than = now() - timedelta(minutes=settings.URL_AUTH_EXPIRE_IN)

            capability = models.AuthToken.objects.get(
                token=token, created_at__gt=later_than
            )
            return capability.user

        except models.AuthToken.DoesNotExist:
            return None


class OAuthAuthBackend(oauth.Mechanism, ModelBackend):

    mechanism_name = 'oauth'


class TVMAuthBackend(tvm.Mechanism, ModelBackend):

    mechanism_name = 'tvm'

    def apply(self, service_ticket, user_ip, user_ticket=None):
        user = super().apply(service_ticket, user_ip, user_ticket)
        if not user.is_authenticated():
            return user
        # Пока не пускаем сервисы без пользователей
        if not user.is_impersonated:
            return self.anonymous()
        try:
            user.fields.update(_get_blackbox_fields(user.uid))
        except PermissionDenied:
            return self.anonymous()
        return user


class ExternalAuthBackend(ModelBackend):
    """
    The original ModelBackend serves external authentication just as well.
    However, being at the end of the backend list, it is called as a last resort
    for INTERNAL authentication. This makes an unnecessary DB query
    trying to with find a user with `username is NULL`.

    To avoid this, we make the external authentication backend
    unsuitable for internal usage by making username and password mandatory.
    """

    # noinspection PyMethodOverriding
    def authenticate(self, request, *, username, password):
        return super().authenticate(
            request, username=username, password=password
        )


def auth_login(request, username, password):

    user = authenticate(request=request, username=username, password=password)

    if user is not None and not user.is_deleted:
        login(request, user)
        request.session[settings.SESSION_KEY_EMAIL] = username
        return True

    return False
