# coding: utf-8

from django.core.exceptions import ObjectDoesNotExist
from django.utils.encoding import force_text
from django.utils.translation import gettext_lazy as _
from startrek_client.exceptions import StartrekError

from procu import jsonschema as js
from procu.api import models
from procu.api.attachment.serializers import AttachmentEntrySerializer
from procu.api.exceptions import FetchUserError, FetchUserNotFound
from procu.api.user.serializers import UserBrief
from procu.api.utils import get_tracker_client, parse_terms
from procu.jsonschema import schema as js
from procu.rest import serializers


class TicketsField(serializers.ListField):
    def __init__(self, **kwargs):
        child = serializers.CharField(
            # 15 (max length of queue id) + 1 (dash) + 8 (up to 10^9-1 tickets)
            max_length=24,
            allow_blank=False,
            error_messages={
                'max_length': _('FIELD_TICKETS::ID_TOO_LONG{max_length}'),
                'blank': _('FIELD_TICKETS::ID_CANNOT_BE_BLANK'),
            },
        )
        if 'label' not in kwargs:
            kwargs['label'] = _('FIELD_TICKETS::LABEL')

        super().__init__(child=child, **kwargs)


class AttachmentField(serializers.PKPrettyField):
    def __init__(self, **kwargs):
        kwargs['queryset'] = models.Attachment.objects.all()
        kwargs['serializer'] = AttachmentEntrySerializer
        kwargs['error_messages'] = {
            'invalid_attachment': _('FIELD_ATTACHMENT::INVALID{pk}')
        }
        super().__init__(**kwargs)

    def to_internal_value(self, data):
        try:
            return self.queryset.get(pk=data)

        except models.Attachment.DoesNotExist:
            self.fail('invalid_attachment', pk=data)

        except (TypeError, ValueError):
            self.fail('incorrect_type', data_type=type(data).__name__)


class TrackerChoiceField(serializers.CharField):
    def __init__(self, *args, **kwargs):
        self.tracker_field = kwargs.pop('field', None)
        super().__init__(*args, **kwargs)

    def get_schema(self, *args, **kwargs):
        schema = super().get_schema(*args, **kwargs)

        options = ['']
        names = {'': '–'}

        if kwargs.get('write', False):
            try:
                client = get_tracker_client()

                field = client.fields.get(self.tracker_field)
                values = field.optionsProvider['values']['LOGIC']

                options.extend(values)
                names.update({v: v for v in values})

            except StartrekError:
                pass

            schema['enum'] = options
            schema['x-names'] = names

        return schema


class EmployeeField(serializers.RelatedField):
    queryset = models.User.objects

    default_error_messages = {
        'fetch_error': _('FIELD_EMPLOYEE::FETCH_ERROR'),
        'not_found': _('FIELD_EMPLOYEE::PROCU_USER_NOT_FOUND'),
        'not_found_username': _('FIELD_EMPLOYEE::EMPLOYEE_NOT_FOUND{username}'),
    }

    def to_internal_value(self, data):

        try:
            data = int(data)
            return models.User.objects.get(id=data)

        except (TypeError, ValueError):
            pass

        except ObjectDoesNotExist:
            self.fail('not_found')

        try:
            return models.User.objects.get_or_create_by_login(username=data)

        except FetchUserNotFound:
            self.fail('not_found_username', username=data)

        except FetchUserError:
            self.fail('fetch_error')

    def to_representation(self, value):
        return UserBrief(value, context=self.context).data

    def get_schema(self, *args, **kwargs):
        write = kwargs.get('write', False)
        if write:
            schema = js.String()
            schema['x-required'] = True
            return schema
        else:
            return UserBrief().meta(*args, **kwargs)


class EnumField(serializers.Field):
    enum_class = None

    default_error_messages = {
        'invalid_value': _('ENUM_FIELD::INVALID_VALUE{given}{options}')
    }

    def __init__(self, **kwargs):
        self.name_only = kwargs.pop('name_only', False)
        self.custom = kwargs.pop('custom', False)

        # ------------------------------------------------

        self.enum = getattr(self, 'enum_class', None)
        if self.enum is None:
            self.enum = kwargs.pop('enum_class', None)

        assert self.enum is not None, 'enum is not specified'

        # ------------------------------------------------

        super().__init__(**kwargs)

    def to_representation(self, value):

        if self.custom and value == '__custom__':
            return {'key': '__custom__', 'name': _('ENUM_FIELD::CUSTOM_VALUE')}

        elif self.name_only:
            return force_text(self.enum.i18n[value])

        return {'key': self.enum.keys[value], 'name': self.enum.i18n[value]}

    def to_internal_value(self, value):

        if self.custom and value == '__custom__':
            return None

        try:
            return self.enum.get_by_key(value)

        except KeyError:
            self.fail(
                'invalid_value',
                given=value,
                options=', '.join(self.enum.keys.values()),
            )

    def get_valid_values(self):
        return list(self.enum.keys)

    def get_schema(self, write=False, *args, **kwargs):
        if write:

            valid_values = self.get_valid_values()

            display_names = {
                self.enum.keys[v]: self.enum.i18n[v] for v in valid_values
            }

            if self.custom:
                display_names['__custom__'] = _('ENUM_FIELD::CUSTOM_VALUE')

            return js.String(
                enum=list(display_names.keys()), **{'x-names': display_names}
            )

        else:
            if self.name_only:
                return js.String()

            return js.Object(('key', js.String()), ('name', js.String()))


class PaymentTermsField(serializers.JSONField):
    def to_internal_value(self, value):
        terms = parse_terms(value)
        return terms

    def get_schema(self, write=False, *args, **kwargs):

        if write:
            return js.String()

        return js.Object(
            ('comment', js.String()),
            ('pre', js.Array(js.Integer())),
            ('post', js.Array(js.Integer())),
        )
