# coding: utf-8

from procu.api import models
from procu.api.utils import dict_diff

PRODUCT_FIELDS = (
    'id',
    'name',
    'qty',
    'price',
    'comment',
    'delivery_time',
    'is_replacement',
    'is_per_unit',
    'currency',
    'is_tax_included',
)


class OfferSnapshot:
    def __init__(self, quote_id, products):
        self.quote_id = quote_id
        self.products = products

    @classmethod
    def create(cls, quote_id) -> 'OfferSnapshot':

        qs = models.Currency.objects.values('id', 'prefix', 'suffix')
        currencies = {c['id']: c for c in qs}

        products = list(
            models.QuoteProduct.objects.filter(quote_id=quote_id)
            .values(*PRODUCT_FIELDS)
            .order_by('id')
        )

        for product in products:
            product['currency'] = currencies[product['currency']]

        return OfferSnapshot(quote_id, products)

    def diff(self, new: 'OfferSnapshot'):
        assert new.quote_id == self.quote_id

        diff = []

        dummy = dict.fromkeys(PRODUCT_FIELDS)

        old_product_map = {product['id']: product for product in self.products}

        for new_product in new.products:
            new_product_id = new_product.get('id')

            if new_product_id in old_product_map:
                # Updated products
                old_product = old_product_map.pop(new_product_id)
                pdiff = dict_diff(old_product, new_product)

                if pdiff:
                    diff.append(pdiff)

            else:
                # Created products
                dummy['is_replacement'] = new_product['is_replacement']
                diff.append(dict_diff(dummy, new_product))

        # Deleted products
        for product in old_product_map.values():
            dummy['is_replacement'] = product['is_replacement']
            diff.append(dict_diff(product, dummy))

        return diff
