# coding: utf-8

import io
from typing import Tuple

import xlsxwriter
from django.utils.translation import gettext as _

from procu.api import models


def export(quote_id: int) -> Tuple[str, io.BytesIO]:

    try:
        request = (
            models.Request.objects.select_related('address', 'legal_entity')
            .prefetch_related('products')
            .get(quotes=quote_id)
        )

    except models.Request.DoesNotExist as exc:
        raise ValueError('Request is not available') from exc

    # --------------------------------------------------------------------------

    output = io.BytesIO()
    workbook = xlsxwriter.Workbook(output, {'in_memory': True})

    TITLE_FMT = workbook.add_format({'bold': True})
    TEXT_FMT = workbook.add_format(
        {'align': 'left', 'valign': 'top', 'text_wrap': 1}
    )

    worksheet = workbook.add_worksheet(_('QUOTE_SUMMARY::SHEET'))

    # --------------------------------------------------------------------------

    headers = (
        _('QUOTE_SUMMARY::PRODUCT_NAME'),
        _('QUOTE_SUMMARY::QTY'),
        _('QUOTE_SUMMARY::COMMENT'),
    )
    worksheet.write_row(0, 0, headers, TITLE_FMT)

    # --------------------------------------------------------------------------

    worksheet.set_column(0, 0, 40)
    worksheet.set_column(2, 2, 60)

    # --------------------------------------------------------------------------

    row_id = 1

    # --------------------------------------------------------------------------
    # Write products

    for row in request.products.all():
        worksheet.write(row_id, 0, row.name)
        worksheet.write(row_id, 1, row.qty)
        worksheet.write(row_id, 2, row.comment)
        row_id += 1

    # --------------------------------------------------------------------------
    # Write description

    if request.description:
        worksheet.write(row_id + 1, 0, _('QUOTE_SUMMARY::COMMENT'), TITLE_FMT)
        worksheet.merge_range(
            row_id + 2,
            0,
            row_id + 2,
            2,
            request.description.replace('\r', ''),
            TEXT_FMT,
        )
        worksheet.set_row(row_id + 2, 30)
        row_id += 3

    # --------------------------------------------------------------------------
    # Write legal entity

    if request.legal_entity:

        pos = (row_id + 1, 0)
        worksheet.write(*pos, _('QUOTE_SUMMARY::LEGAL_ENTITY'), TITLE_FMT)

        pos = (row_id + 2, 0, row_id + 2, 2)
        worksheet.merge_range(*pos, request.legal_entity.title, TEXT_FMT)

        worksheet.set_row(row_id + 2, 30)
        row_id += 3

        pos = (row_id + 1, 0)
        worksheet.write(
            *pos, _('QUOTE_SUMMARY::LEGAL_ENTITY_DETAILS'), TITLE_FMT
        )

        pos = (row_id + 2, 0, row_id + 2, 2)
        text = request.legal_entity.details.replace('\r', '')
        worksheet.merge_range(*pos, text, TEXT_FMT)

        worksheet.set_row(row_id + 2, 60)
        row_id += 3

    # --------------------------------------------------------------------------
    # Write address

    worksheet.write(row_id + 1, 0, _('QUOTE_SUMMARY::ADDRESS'), TITLE_FMT)

    text = request.address.text.replace('\r', '')
    pos = (row_id + 2, 0, row_id + 2, 2)
    worksheet.merge_range(*pos, text, TEXT_FMT)

    worksheet.set_row(row_id + 2, 30)

    # --------------------------------------------------------------------------

    workbook.close()

    return (f'enquiry_YP{request.enquiry_id}.xlsx', output)
