# coding: utf-8

import re
from functools import partial
from typing import Tuple

from django.contrib.auth.models import Group
from django.core import mail
from django.utils.translation import gettext_lazy as _
from rest_framework import exceptions, validators

from procu.api import models
from procu.api.push.external import notify_new_user
from procu.api.supplier.serializers import SupplierBrief
from procu.rest import serializers
from procu.rest.fields import PKPrettyField


class AgentInfo(serializers.Serializer):

    position = serializers.CharField(
        max_length=255,
        label=_('CONTACT_INFO::POSITION'),
        required=False,
        allow_blank=False,
    )
    phone_landline = serializers.CharField(
        max_length=63,
        label=_('CONTACT_INFO::PHONE_LANDLINE'),
        required=False,
        allow_blank=False,
    )
    phone_mobile = serializers.CharField(
        max_length=63,
        label=_('CONTACT_INFO::PHONE_MOBILE'),
        required=False,
        allow_blank=False,
    )
    work_address = serializers.CharField(
        label=_('CONTACT_INFO::WORK_ADDRESS'), required=False, allow_blank=False
    )

    class Meta:
        fields = ('position', 'phone_landline', 'phone_mobile', 'work_address')


# ------------------------------------------------------------------------------


class Fields(object):

    email = partial(
        serializers.EmailField,
        validators=[
            validators.UniqueValidator(
                queryset=models.User.objects.all(),
                message=_('CONTACTS::ERROR_USER_EXISTS'),
                lookup='iexact',
            )
        ],
        max_length=127,
    )

    full_name = partial(serializers.CharField, read_only=True)

    supplier = partial(
        PKPrettyField, serializer=SupplierBrief, required=False, read_only=True
    )

    info = partial(AgentInfo, required=False)


# ------------------------------------------------------------------------------


class Create(serializers.ModelSerializer):
    check_for_nested_writes = False

    email = Fields.email()
    info = Fields.info()

    full_name = serializers.CharField(
        allow_blank=True,
        required=True,
        max_length=255,
        label=_('CONTACTS::USER_FULL_NAME'),
        style={'x-hint': _('CONTACTS::USER_FIRST_NAME_LAST_NAME')},
    )

    class Meta:
        model = models.User
        fields = ('email', 'full_name', 'comment', 'info')

    @staticmethod
    def parse_name(text: str) -> Tuple[str, str]:

        text = text.strip()

        if text == '':
            return ('', '')

        m = re.match(r'^(\S+)\s+(\S+)$', text)
        if m:
            return m.groups()

        m = re.match(r'^(\S+)$', text)
        if m:
            return (m.group(0), '')

        raise ValueError('Could not parse name')

    def validate(self, data):
        full_name = data.pop('full_name', '')

        try:
            data['first_name'], data['last_name'] = self.parse_name(full_name)

        except ValueError:
            raise exceptions.ValidationError(
                {'full_name': _('CONTACTS::ERROR_INVALID_NAME')}
            )

        return data

    def create(self, data):
        user = super().create(data)

        with mail.get_connection(fail_silently=True) as conn:
            conn.send_messages(notify_new_user(user))

        return user


# ------------------------------------------------------------------------------


class Update(serializers.ModelSerializer):
    check_for_nested_writes = False

    email = Fields.email()
    first_name = serializers.CharField(max_length=255, allow_blank=True)
    last_name = serializers.CharField(max_length=255, allow_blank=True)
    info = Fields.info()

    class Meta:
        model = models.User
        fields = ('email', 'first_name', 'last_name', 'comment', 'info')


# ------------------------------------------------------------------------------


class List(serializers.ModelSerializer):

    email = Fields.email()
    full_name = Fields.full_name()
    supplier = Fields.supplier()

    class Meta:
        model = models.User
        fields = (
            'id',
            'email',
            'supplier',
            'full_name',
            'first_name',
            'last_name',
            'comment',
            'is_deleted',
        )


# ------------------------------------------------------------------------------


class Retrieve(serializers.ModelSerializer):

    email = Fields.email()
    full_name = Fields.full_name()
    info = Fields.info()
    supplier = Fields.supplier()

    class Meta:
        model = models.User
        fields = (
            'id',
            'email',
            'info',
            'supplier',
            'full_name',
            'first_name',
            'last_name',
            'comment',
            'is_deleted',
        )
        extra_kwargs = {'is_deleted': {'label': _('CONTACTS::IS_DELETED')}}
