# coding: utf-8

from django.utils.translation import gettext_lazy as _

from procu import jsonschema as js
from procu.api import models
from procu.api.common.fields import EmployeeField, UserBrief
from procu.api.enums import ACCESS_SOURCE
from procu.rest import serializers


def is_admin_or_manager(user):
    return user is not None and user.has_perm('api.progress_enquiry')


def is_author_or_assignee(instance):
    return instance is not None and bool(
        {ACCESS_SOURCE.AUTHOR, ACCESS_SOURCE.ASSIGNEE} & set(instance.sources)
    )


class AllowQuoteComments(serializers.BooleanField):
    def __init__(self, *args, **kwargs):
        kw = dict(kwargs)
        kw['source'] = '*'
        super().__init__(*args, **kw)

    def to_representation(self, instance):
        return (
            is_admin_or_manager(instance.user) or instance.allow_quote_comments
        )

    def to_internal_value(self, data):
        assert self.parent.instance
        instance = self.parent.instance

        if is_admin_or_manager(instance.user):
            return {'allow_quote_comments': False}

        return {'allow_quote_comments': data}

    def get_schema(self, *args, **kwargs):
        kw = {}
        instance = self.parent.instance

        if instance is not None and is_admin_or_manager(instance.user):
            kw['x-disabled'] = True
        return js.Boolean(**kw)


class SubscriptionField(serializers.BooleanField):
    def __init__(self, *args, **kwargs):
        kw = dict(kwargs)
        kw['source'] = '*'
        super().__init__(*args, **kw)

    def to_internal_value(self, data):
        assert self.parent.instance
        instance = self.parent.instance

        if is_author_or_assignee(instance):
            return {'is_subscribed': True}
        return {'is_subscribed': data}

    def get_schema(self, *args, **kwargs):
        kw = {}
        instance = self.parent.instance
        if is_author_or_assignee(instance):
            kw['x-disabled'] = True
        return js.Boolean(**kw)


class ListAdmin(serializers.ModelSerializer):
    user = UserBrief()
    can_update = serializers.SerializerMethodField()
    allow_quote_comments = AllowQuoteComments()

    class Meta:
        model = models.EnquiryAccess
        fields = (
            'id',
            'user',
            'allow_quote_comments',
            'is_subscribed',
            'can_update',
        )

    def get_can_update(self, instance):
        can_update = getattr(self.context['view'], 'can_update')
        return bool(can_update)


class ListNonAdmin(ListAdmin):
    user = UserBrief()
    can_update = serializers.SerializerMethodField()

    class Meta:
        model = models.EnquiryAccess
        fields = ('id', 'user', 'is_subscribed', 'can_update')


class Create(serializers.Serializer):
    users = EmployeeField(
        required=True, many=True, label=_('ENQUIRY_ACCESS::USERS')
    )


class Retrieve(serializers.ModelSerializer):
    user = UserBrief()
    allow_quote_comments = AllowQuoteComments()

    class Meta:
        model = models.EnquiryAccess
        fields = ('id', 'user', 'allow_quote_comments', 'is_subscribed')


class Update(serializers.ModelSerializer):
    allow_quote_comments = AllowQuoteComments()
    is_subscribed = SubscriptionField()

    class Meta:
        model = models.EnquiryAccess
        fields = ('allow_quote_comments', 'is_subscribed')
