# coding: utf-8

from rest_framework import status
from rest_framework.response import Response

from procu.api import models
from procu.api.enums import ACCESS_SOURCE
from procu.api.utils import is_readonly, json_dumps
from procu.rest import generics, pagination
from . import serializers
from .permissions import AccessPermissions
from .timeline import AccessSnapshot
from .utils import give_access_to_user


class AccessList(generics.ListCreateAPIView):
    permission_classes = (AccessPermissions,)
    pagination_class = pagination.NoPagination

    def post_initial(self, request):
        self.can_update = not is_readonly() and request.user.has_perm(
            'api.progress_enquiry'
        )

    def get_serializer_class(self):
        if self.request.method == 'POST':
            return serializers.Create
        else:
            if self.request.user.has_perm('api.progress_enquiry'):
                return serializers.ListAdmin
            else:
                return serializers.ListNonAdmin

    def get_queryset(self):
        return (
            models.EnquiryAccess.objects.filter(
                enquiry=self.kwargs['enquiry_id'],
                sources__contains=[ACCESS_SOURCE.ACCESS],
            )
            .select_related('user')
            .order_by('id')
        )

    def post(self, request, *args, **kwargs):

        enquiry_id = self.kwargs['enquiry_id']

        old = AccessSnapshot.create(enquiry_id)

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data

        for user in data['users']:
            give_access_to_user(enquiry_id, user)

        new = AccessSnapshot.create(enquiry_id)

        diff = old.diff(new)

        if diff:
            models.Log.objects.create(
                enquiry_id=enquiry_id,
                user=self.request.user,
                type='update_access',
                data=json_dumps(diff),
            )

        return Response({'result': 'ok'})


class AccessRetrieveUpdateDestroy(generics.RetrieveUpdateDestroyAPIView):
    permission_classes = (AccessPermissions,)
    serializer_class = serializers.Update
    lookup_url_kwarg = 'access_id'

    def get_serializer_class(self):
        if self.request.method == 'PATCH':
            return serializers.Update
        else:
            return serializers.Retrieve

    def get_queryset(self):
        return models.EnquiryAccess.objects.filter(
            enquiry=self.kwargs['enquiry_id'],
            sources__contains=[ACCESS_SOURCE.ACCESS],
        ).order_by('id')

    def destroy(self, request, *args, **kwargs):
        obj = self.get_object()

        obj.sources = list({*obj.sources} - {ACCESS_SOURCE.ACCESS})

        if not obj.sources:
            obj.delete()
        else:
            perm = obj.user.has_perm('api.progress_enquiry')
            obj.allow_quote_comments = perm
            obj.is_subscribed = False

            obj.save(update_fields=['sources'])

        return Response(status=status.HTTP_204_NO_CONTENT)

    def dispatch(self, request, *args, **kwargs):

        is_loggable = request.method in ('PATCH', 'DELETE')

        enquiry_id = self.kwargs['enquiry_id']

        if is_loggable:
            old = AccessSnapshot.create(enquiry_id)

        response = super().dispatch(request, *args, **kwargs)

        if is_loggable:
            new = AccessSnapshot.create(enquiry_id)
            diff = old.diff(new)

            if diff:
                models.Log.objects.create(
                    enquiry_id=enquiry_id,
                    user=self.request.user,
                    type='update_access',
                    data=json_dumps(diff),
                )

        return response
