# coding: utf-8

from django.core import mail
from django.utils.translation import gettext as _

from procu.api import models
from procu.api.enquiry.access.utils import give_access_to_user
from procu.api.push.internal import notify_enquiry_comment
from procu.api.utils import dict_diff, json_dumps
from procu.rest.exceptions import ValidationError
from .serializers import Create


def create_enquiry_comment(
    enquiry_id, user, data, invitees, attachment_ids=(), notify=True
):

    if not models.Enquiry.objects.filter(id=enquiry_id).exists():
        raise ValueError("`enquiry_id` is invalid")

    if not (data.get('message') or attachment_ids):
        raise ValidationError({'message': _('COMMENTS::ERROR_BLANK_NO_FILES')})

    # ----------------------------------------------------------
    # Create

    kwargs = {'enquiry_id': enquiry_id, 'author_id': user.id, **data}

    comment = models.EnquiryComment.objects.create(**kwargs)
    models.Attachment.objects.filter(
        author=user, id__in=attachment_ids, enquiry_comment__isnull=True
    ).update(enquiry_comment=comment)

    comment.invitees.add(*invitees)

    # Add invitees to access list if needed
    for invitee in invitees:
        give_access_to_user(enquiry_id, invitee)

    # ----------------------------------------------------------
    # Timeline

    new = Create(comment).data
    diff = dict_diff({}, new)

    models.Log.objects.create(
        user=user,
        enquiry_id=enquiry_id,
        type='create_enquiry_comment',
        data=json_dumps(diff),
        new=json_dumps(new),
    )

    # ----------------------------------------------------------
    # Notifications

    if notify:
        messages = notify_enquiry_comment(comment.id, user)

        with mail.get_connection(fail_silently=True) as conn:
            conn.send_messages(messages)

    return comment
