# coding: utf-8
from functools import partial

from django.utils.encoding import force_text
from django.utils.translation import gettext_lazy as _

from procu.api import models
from procu.api.address.serializers import AddressSerializer
from procu.api.category.serializers import EnquiryCategoryBriefSerializer
from procu.api.common.fields import TicketsField, EnumField, AttachmentField
from procu.api.enums import ER, ES, PRIORITY, STATE
from procu.api.legal_entity.serializers import LegalEntityBrief
from procu.api.oebs.serializers import (
    CFOBrief,
    ProgramBrief,
    ProjectBrief,
    TaskBrief,
    MVPBrief,
    ServiceBrief,
    BudgetLineBrief,
    SystemBrief,
    SubSystemBrief,
)
from procu.api.user.serializers import UserBrief
from procu.jsonschema import schema as js
from procu.rest import serializers
from procu.rest.fields import PKPrettyField


class EStatusField(serializers.ReadOnlyField):
    def __init__(self, **kwargs):
        self.name_only = kwargs.pop('name_only', False)
        super().__init__(**kwargs)

    def to_representation(self, value):

        if self.name_only:
            return force_text(ES.i18n[value])

        return {'key': ES.keys[value], 'name': ES.i18n[value]}

    def get_schema(self, *args, **kwargs):
        if self.name_only:
            return js.String()

        return js.Object(('key', js.String()), ('name', js.String()))


class EReasonField(serializers.ReadOnlyField):
    def __init__(self, **kwargs):
        self.name_only = kwargs.pop('name_only', False)
        super().__init__(**kwargs)

    def to_representation(self, value):

        if self.name_only:
            return force_text(ER.i18n[value])

        return {'key': ER.keys[value], 'name': ER.i18n[value]}

    def get_schema(self, *args, **kwargs):

        if self.name_only:
            return js.String()

        return js.Object(('key', js.String()), ('name', js.String()))


class OracleFields(object):

    cfo = partial(
        PKPrettyField,
        queryset=models.OracleCFO.objects.all(),
        serializer=CFOBrief,
        allow_null=True,
        required=False,
    )

    program = partial(
        PKPrettyField,
        queryset=models.OracleProgram.objects.all(),
        serializer=ProgramBrief,
        allow_null=True,
        required=False,
    )

    project = partial(
        PKPrettyField,
        queryset=models.OracleProject.objects.all(),
        serializer=ProjectBrief,
        allow_null=True,
        required=False,
    )

    task = partial(
        PKPrettyField,
        queryset=models.OracleTask.objects.all(),
        serializer=TaskBrief,
        allow_null=True,
        required=False,
    )

    mvp = partial(
        PKPrettyField,
        queryset=models.OracleMVP.objects.all(),
        serializer=MVPBrief,
        allow_null=True,
        required=False,
    )

    service = partial(
        PKPrettyField,
        queryset=models.OracleService.objects.all(),
        serializer=ServiceBrief,
        allow_null=True,
        required=False,
    )

    budget_line = partial(
        PKPrettyField,
        queryset=models.OracleBudgetLine.objects.all(),
        serializer=BudgetLineBrief,
        allow_null=True,
        required=False,
    )

    system = partial(
        PKPrettyField,
        queryset=models.OracleSystem.objects.all(),
        serializer=SystemBrief,
        allow_null=True,
        required=False,
    )

    subsystem = partial(
        PKPrettyField,
        queryset=models.OracleSubSystem.objects.all(),
        serializer=SubSystemBrief,
        allow_null=True,
        required=False,
    )


class Product(serializers.Serializer):
    name = serializers.CharField(
        allow_blank=False, label=_('ENQUIRY_PRODUCT::NAME'), max_length=512
    )
    qty = serializers.IntegerField(min_value=0, label=_('ENQUIRY_PRODUCT::QTY'))
    comment = serializers.CharField(
        allow_blank=True, label=_('ENQUIRY_PRODUCT::COMMENT')
    )


class Fields(object):
    """
    A collection of field callables with common predefined parameters
    """

    reference = partial(TicketsField, required=False, allow_empty=True)

    status = partial(EStatusField)
    reason = partial(EReasonField)
    priority = partial(EnumField, enum_class=PRIORITY, required=False)

    category = partial(
        PKPrettyField,
        queryset=models.EnquiryCategory.objects.all(),
        serializer=EnquiryCategoryBriefSerializer,
        allow_null=True,
        required=False,
    )

    author = partial(PKPrettyField, serializer=UserBrief, read_only=True)

    manager = partial(
        PKPrettyField,
        queryset=models.User.objects.filter(is_staff=True),
        serializer=UserBrief,
        allow_null=True,
        required=False,
        label=_('MODEL_ENQUIRY::MANAGER'),
    )

    assignee = partial(manager, source='manager')

    legal_entity = partial(
        PKPrettyField,
        queryset=models.LegalEntity.objects.all(),
        serializer=LegalEntityBrief,
        allow_null=True,
    )

    address = partial(
        PKPrettyField,
        queryset=models.Address.objects.all(),
        serializer=AddressSerializer,
        allow_null=True,
    )

    attachments = partial(
        AttachmentField,
        label=_('ENQUIRIES::ATTACHMENTS'),
        many=True,
        required=False,
    )
    deadline_at = partial(
        serializers.DateTimeField, required=False, allow_null=True
    )
    due_at = partial(serializers.DateField, required=False, allow_null=True)

    products = partial(
        Product,
        label=_('ENQUIRIES::PRODUCTS'),
        allow_empty=True,
        many=True,
        source='initial_products',
    )

    description = partial(
        serializers.CharField,
        source='internal_comment',
        label=_('MODEL_ENQUIRY::INTERNAL_COMMENT'),
        allow_blank=True,
    )

    state = partial(EnumField, enum_class=STATE, required=False)
