# coding: utf-8

from django.utils.translation import gettext_lazy as _
from rest_framework.exceptions import ValidationError
from rest_framework.fields import get_attribute

from procu.api import enums, models
from procu.api.common.fields import EnumField
from procu.api.currency.serializers import CurrencyBriefSerializer
from procu.api.unit.serializers import UnitSerializer
from procu.api.utils.decimal import money
from procu.rest import serializers
from .fields import EnumPK


def generalize(items, field):
    s = set(get_attribute(p, (field,)) for p in items)
    if len(s) == 1:
        return s.pop()
    else:
        return '__custom__'


class AddressSerializer(serializers.ModelSerializer):
    class Meta:
        model = models.Address
        fields = ('id', 'label')


class LegalEntitySerializer(serializers.ModelSerializer):
    class Meta:
        model = models.LegalEntity
        fields = ('id', 'title')


class QueryParams(serializers.Serializer):
    products = serializers.ListField(child=serializers.IntegerField())
    currency = serializers.PrimaryKeyRelatedField(
        queryset=models.Currency.objects.filter(is_deleted=False)
    )
    tax = serializers.BooleanField(default=True)
    rounded = serializers.BooleanField(default=False)


class InvoiceProductUpdateInternal(serializers.ModelSerializer):

    price_for = EnumField(enum_class=enums.PRICE)

    address = EnumPK(
        queryset=models.Address.objects.filter(is_deleted=False),
        display_field='label',
        serializer=AddressSerializer,
    )

    class Meta:
        model = models.InvoiceProduct
        fields = (
            'price',
            'price_for',
            'address',
            'delivery_expected_at',
            'delivery_happened_at',
        )
        extra_kwargs = {
            'delivery_expected_at': {
                'label': _('INVOICE_FORM::DELIVERY_EXPECTED_AT')
            },
            'delivery_happened_at': {
                'label': _('INVOICE_FORM::DELIVERY_HAPPENED_AT')
            },
            'address': {'label': _('INVOICE_FORM::ADDRESS')},
        }


class InvoiceProductUpdateExternal(serializers.ModelSerializer):

    price_for = EnumField(enum_class=enums.PRICE)

    class Meta:
        model = models.InvoiceProduct
        fields = ('price', 'price_for', 'delivery_expected_at')
        extra_kwargs = {
            'delivery_happened_at': {
                'label': _('INVOICE_FORM::DELIVERY_HAPPENED_AT')
            }
        }


class InvoiceProductRaw(serializers.ModelSerializer):
    tax_mode = EnumField(enum_class=enums.TAX, label=' ')
    price_for = EnumField(enum_class=enums.PRICE, label=' ')
    currency = EnumPK(
        queryset=models.Currency.objects.filter(is_deleted=False),
        display_field='char_code',
        serializer=CurrencyBriefSerializer,
        label=' ',
    )
    unit = EnumPK(
        queryset=models.Unit.objects.filter(is_deleted=False),
        display_field='shortname',
        serializer=UnitSerializer,
        label=' ',
    )
    address = EnumPK(
        queryset=models.Address.objects.filter(is_deleted=False),
        display_field='label',
        serializer=AddressSerializer,
        label=' ',
    )

    class Meta:
        model = models.InvoiceProduct
        fields = (
            'id',
            'name',
            'qty',
            'unit',
            'address',
            'price',
            'currency',
            'price_for',
            'tax',
            'tax_mode',
            'delivery_expected_at',
            'delivery_happened_at',
        )
        extra_kwargs = {
            'name': {'max_length': 255},
            'qty': {'style': {'x-required': True}},
            'price': {'style': {'x-required': True}},
            'tax': {'default': money('18.00')},
            'delivery_expected_at': {
                'label': _('INVOICE_FORM::DELIVERY_EXPECTED_AT')
            },
            'delivery_happened_at': {
                'label': _('INVOICE_FORM::DELIVERY_HAPPENED_AT')
            },
        }


class InvoiceProduct(InvoiceProductRaw):
    def to_representation(self, instance):
        data = super().to_representation(instance)

        if not self.context.get('can_see_prices', False):
            data['price'] = '****'
            data['total'] = '****'

        else:
            try:
                if instance.price_for == enums.PRICE.TOTAL:
                    data['price'] = money(
                        instance.price / instance.qty, ndigits=2
                    )
                    data['total'] = instance.price
                else:
                    data['total'] = money(
                        instance.price * instance.qty, ndigits=2
                    )

            except AttributeError:
                pass

        data['can_update'] = self.context.get('can_update', False)

        return data


class InvoiceRetrieve(serializers.ModelSerializer):

    products = InvoiceProduct(label=_('INVOICE_FORM::PRODUCTS'), many=True)
    created_at = serializers.DateTimeField(read_only=True)
    legal_entity = LegalEntitySerializer()

    class Meta:
        model = models.Invoice
        fields = ('id', 'products', 'created_at', 'legal_entity')

    def to_representation(self, instance):
        data = super().to_representation(instance)

        for seq_id, product in enumerate(data['products'], 1):
            product['seq_id'] = seq_id

        return data


class InvoiceUpdate(serializers.ModelSerializer):

    legal_entity = serializers.PrimaryKeyRelatedField(
        queryset=models.LegalEntity.objects.filter(is_deleted=False),
        label=_('INVOICE_FORM::LEGAL_ENTITY'),
        allow_null=False,
        required=True,
    )

    class Meta:
        model = models.Invoice
        fields = ('legal_entity',)


class InvoiceCreate(serializers.Serializer):

    id = serializers.IntegerField(required=False)

    supplier = serializers.PrimaryKeyRelatedField(
        queryset=models.Supplier.objects.filter(is_deleted=False),
        label=_('INVOICE_FORM::SUPPLIER'),
        style={'x-hint': _('INVOICE_FORM::QUOTE_HINT')},
    )

    legal_entity = serializers.PrimaryKeyRelatedField(
        queryset=models.LegalEntity.objects.filter(is_deleted=False),
        label=_('INVOICE_FORM::LEGAL_ENTITY'),
        required=False,
    )

    products = InvoiceProduct(label=_('INVOICE_FORM::PRODUCTS'), many=True)
    created_at = serializers.DateTimeField(read_only=True)

    price_for = EnumField(
        enum_class=enums.PRICE_COMMON,
        label=_('MODEL_INVOICEPRODUCT::PRICE_FOR'),
        custom=True,
    )
    tax_mode = EnumField(
        enum_class=enums.TAX_COMMON,
        label=_('MODEL_INVOICEPRODUCT::TAX_MODE'),
        custom=False,
    )
    currency = EnumPK(
        queryset=models.Currency.objects.filter(is_deleted=False),
        display_field='char_code',
        serializer=CurrencyBriefSerializer,
        label=_('MODEL_INVOICEPRODUCT::CURRENCY'),
        custom=False,
    )
    unit = EnumPK(
        queryset=models.Unit.objects.filter(is_deleted=False),
        display_field='shortname',
        serializer=UnitSerializer,
        label=_('MODEL_INVOICEPRODUCT::UNIT'),
        custom=True,
    )

    address = EnumPK(
        queryset=models.Address.objects.filter(is_deleted=False),
        display_field='label',
        serializer=AddressSerializer,
        label=_('MODEL_INVOICEPRODUCT::ADDRESS'),
        custom=True,
    )

    def validate(self, data):

        for product in data['products']:
            for field in (
                'price_for',
                'tax_mode',
                'currency',
                'unit',
                'address',
            ):
                if data.get(field) is not None:
                    product[field] = data[field]

        return data


class Create(serializers.Serializer):
    invoices = InvoiceCreate(many=True)

    def create(self, validated_data):

        objs = []

        for invoice in validated_data['invoices']:

            products = invoice.pop('products')

            for field in (
                'price_for',
                'tax_mode',
                'currency',
                'unit',
                'address',
            ):
                invoice.pop(field)

            invoice = models.Invoice.objects.create(
                enquiry=validated_data.get('enquiry'), **invoice
            )
            objs.append(invoice)

            for product in products:
                models.InvoiceProduct.objects.create(invoice=invoice, **product)

        return objs
