# coding: utf-8

from django.db.models import F

from procu.api import models
from procu.api.enums import PRICE, TAX
from procu.api.utils import dict_diff, format_price

PRODUCT_FIELDS = {
    'id': None,
    'name': None,
    'qty': None,
    'price': None,
    'price_for': None,
    'tax_mode': None,
    'delivery_expected_at': None,
    'delivery_happened_at': None,
    'address': F('address__label'),
    'unit': F('unit__shortname'),
}

INVOICE_FIELDS = ('id', 'created_at', 'supplier', 'legal_entity')


class InvoiceSnapshot:
    def __init__(self, invoice, products):
        self.invoice = invoice
        self.products = products

    @classmethod
    def create(cls, invoice_id):
        invoice = (
            models.Invoice.objects.values('id', 'created_at')
            .annotate(
                supplier=F('supplier__title'),
                legal_entity=F('legal_entity__title'),
            )
            .get(id=invoice_id)
        )

        # ----------------------------------------------

        qs = (
            models.InvoiceProduct.objects.filter(invoice_id=invoice_id)
            .values(*(k for k, v in PRODUCT_FIELDS.items() if v is None))
            .annotate(
                prefix=F('currency__prefix'),
                suffix=F('currency__suffix'),
                **{k: v for k, v in PRODUCT_FIELDS.items() if v},
            )
        )

        products = []

        for product in qs:
            product['price_for'] = PRICE.i18n[product['price_for']]
            product['tax_mode'] = TAX.i18n[product['tax_mode']]

            product['price'] = '{}\u00A0{}\u00A0{}'.format(
                product['prefix'],
                format_price(product['price']),
                product['suffix'],
            ).strip()

            product.pop('prefix', None)
            product.pop('suffix', None)

            products.append(product)

        return InvoiceSnapshot(invoice, products)

    def diff_create(self):

        dummy_invoice = dict.fromkeys(INVOICE_FIELDS)

        diff = dict_diff(dummy_invoice, self.invoice)
        diff['products'] = []

        dummy = dict.fromkeys(PRODUCT_FIELDS)

        for product in self.products:
            diff['products'].append(dict_diff(dummy, product))

        return diff

    def diff_destroy(self):

        dummy_invoice = dict.fromkeys(INVOICE_FIELDS)

        diff = dict_diff(self.invoice, dummy_invoice)
        diff['products'] = []

        dummy = dict.fromkeys(PRODUCT_FIELDS)

        for product in self.products:
            diff['products'].append(dict_diff(product, dummy))

        return diff

    def diff_update(self, new: 'InvoiceSnapshot'):
        assert new.invoice['id'] == self.invoice['id']

        if self.products == new.products and self.invoice == new.invoice:
            return {}

        diff = dict_diff(self.invoice, new.invoice, show_unchanged=True)
        if not diff:
            diff.update(new.invoice)
        diff['products'] = []

        dummy = dict.fromkeys(PRODUCT_FIELDS)

        old_product_map = {product['id']: product for product in self.products}

        for new_product in new.products:
            new_product_id = new_product.get('id')

            if new_product_id in old_product_map:
                # Updated product
                old_product = old_product_map.pop(new_product_id)
                pdiff = dict_diff(old_product, new_product)
                if pdiff:
                    diff['products'].append(pdiff)

            else:
                # Created product
                diff['products'].append(dict_diff(dummy, new_product))

        # Deleted products
        for product in old_product_map.values():
            diff['products'].append(dict_diff(product, dummy))

        return diff
