# coding: utf-8

from collections import defaultdict

from rest_framework import status
from rest_framework.response import Response

from procu.api import enums, models
from procu.api.enums import QS
from procu.api.utils import is_readonly, json_dumps
from procu.rest import generics, pagination, permissions
from . import serializers
from .permissions import CheckoutPermission
from .timeline import InvoiceSnapshot


class Invoice(generics.ListCreateAPIView):
    permission_classes = (CheckoutPermission,)
    pagination_class = pagination.NoPagination
    serializer_class = serializers.Create
    query_params_serializer_class = serializers.QueryParams
    lookup_url_kwarg = 'enquiry_id'

    def get_serializer_context(self):
        context = super().get_serializer_context()
        context['can_see_prices'] = True
        return context

    def get_queryset(self):
        return models.Enquiry.objects.permitted(self.request.user)

    def post_initial(self, request):
        enquiry = self.enquiry = self.get_object()
        self.check_object_permissions(request, enquiry)

    def get(self, request, *args, **kwargs):

        rfx = models.Request.objects.get(enquiry_id=self.kwargs['enquiry_id'])

        params = self.query_params

        qs = (
            models.QuoteProduct.objects.prepared(totals=False, **params)
            .filter(enquiry_product__request=rfx, is_deleted=False)
            .select_related('enquiry_product')
        )

        qps = qs.filter(id__in=params['products'])

        qps_by_quotes = defaultdict(list)
        for qp in qps:
            qps_by_quotes[qp.quote_id].append(qp)

        quotes = {
            q.id: q
            for q in models.Quote.objects.filter(request=rfx).select_related(
                'supplier'
            )
        }

        unit = (
            models.Unit.objects.filter(is_deleted=False).order_by('id').first()
        )

        address = self.enquiry.address
        legal_entity = self.enquiry.legal_entity

        invoices = []

        for quote_id, qps in qps_by_quotes.items():

            products = []

            for qp in qps:

                product = {
                    'name': qp.name,
                    'qty': qp.qty,
                    'tax_mode': (
                        enums.TAX.INCLUDED
                        if params['tax']
                        else enums.TAX.EXCLUDED
                    ),
                    'tax': qp.tax,
                    'price': (qp.value if qp.is_per_unit else qp.total),
                    'price_for': (
                        enums.PRICE.UNIT
                        if qp.is_per_unit
                        else enums.PRICE.TOTAL
                    ),
                    'currency': params['currency'],
                    'unit': unit,
                    'address': address,
                }

                products.append(product)

            invoice = {
                'quote': quotes[quote_id],
                'supplier': quotes[quote_id].supplier,
                'legal_entity': legal_entity,
                'products': products,
                # --------------------------------------------
                'tax_mode': serializers.generalize(products, 'tax_mode'),
                'currency': serializers.generalize(products, 'currency'),
                'unit': serializers.generalize(products, 'unit'),
                'price_for': serializers.generalize(products, 'price_for'),
                'address': serializers.generalize(products, 'address'),
            }

            invoices.append(invoice)

        return Response(self.get_serializer({'invoices': invoices}).data)

    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        objs = serializer.save(enquiry=self.enquiry)

        for obj in objs:
            diff = InvoiceSnapshot.create(obj.id).diff_create()
            models.Log.objects.create(
                enquiry_id=self.enquiry.id,
                user=self.request.user,
                type='create_invoice',
                data=json_dumps(diff),
            )

        serializer = serializers.InvoiceRetrieve(objs, many=True)

        return Response(serializer.data, status=status.HTTP_201_CREATED)


class InvoiceRetrieveUpdateDestroy(generics.RetrieveUpdateDestroyAPIView):
    permission_classes = (permissions.EntryPermission,)
    serializer_class = serializers.InvoiceRetrieve
    lookup_url_kwarg = 'invoice_id'

    def get_serializer_class(self):
        return {
            'PATCH': serializers.InvoiceUpdate,
            'GET': serializers.InvoiceRetrieve,
        }.get(self.request.method, self.serializer_class)

    def get_serializer_context(self):
        user = self.request.user
        invoice = self.object
        enquiry = invoice.enquiry

        context = super().get_serializer_context()

        if user.is_staff:
            summary_perm = 'api.summary_enquiry'
            can_see_prices = summary_perm in enquiry.permissions(user)
        else:
            can_see_prices = True

        context['can_see_prices'] = can_see_prices

        context['can_update'] = (
            not is_readonly()
            and (user.is_staff or invoice.quote.status < QS.CLOSED)
            and user.has_perm('api.update_invoiceproduct')
        )

        return context

    def perform_update(self, serializer):
        old = InvoiceSnapshot.create(self.kwargs['invoice_id'])
        super().perform_update(serializer)

        invoice = serializer.instance

        new = InvoiceSnapshot.create(self.kwargs['invoice_id'])
        diff = old.diff_update(new)
        if diff:
            models.Log.objects.create(
                enquiry_id=invoice.enquiry_id,
                user=self.request.user,
                type='update_invoice',
                data=json_dumps(diff),
            )

    def perform_destroy(self, instance):
        diff = InvoiceSnapshot.create(instance.id).diff_destroy()
        models.Log.objects.create(
            enquiry_id=self.kwargs['enquiry_id'],
            user=self.request.user,
            type='remove_invoice',
            data=json_dumps(diff),
        )
        super().perform_destroy(instance)

    def get_queryset(self):
        return models.Invoice.objects.permitted(
            self.request.user
        ).prefetch_related('products', 'products__currency', 'products__unit')
