# coding: utf-8

from django.http import Http404
from django.utils.timezone import now
from rest_framework import response

from procu.api import models
from procu.rest import generics
from procu.rest.permissions import StaffOnly
from . import serializers


class NoteUpdate(generics.RetrieveUpdateAPIView):
    permission_classes = (StaffOnly,)
    serializer_class = serializers.RetrieveUpdate

    lookup_field = 'enquiry_id'
    lookup_url_kwarg = 'enquiry_id'

    def post_initial(self, request):
        enquiry = self.enquiry = generics.get_object_or_404(
            models.Enquiry.objects.permitted(self.request.user).only('id'),
            pk=self.kwargs['enquiry_id'],
        )
        self.check_object_permissions(request, enquiry)

    def get_queryset(self):
        return models.EnquiryNote.objects.permitted(self.request.user)

    def get_object(self):
        try:
            return super().get_object()

        except Http404:
            return models.EnquiryNote(updated_at=None, created_at=None)

    def update(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(
            instance, data=request.data, partial=True
        )
        serializer.is_valid(raise_exception=True)

        instance, _ = models.EnquiryNote.objects.update_or_create(
            user=self.request.user,
            enquiry=self.enquiry,
            defaults={
                'text': serializer.validated_data['text'],
                'updated_at': now(),
            },
        )

        return response.Response(self.get_serializer(instance).data)
