# coding: utf-8
from django.conf import settings
from django.utils.translation import gettext_lazy as _
from rest_framework.exceptions import ValidationError
from rest_framework.response import Response

from procu.api import models
from procu.api.utils import json_dumps
from procu.rest import generics
from ..permissions import PublishRequestPermission


class DraftView(generics.GenericAPIView):
    permission_classes = (PublishRequestPermission,)
    lookup_url_kwarg = 'enquiry_id'

    def get_queryset(self):
        return models.Enquiry.objects.permitted(self.request.user)

    def post_initial(self, request):
        enquiry = self.enquiry = generics.get_object_or_404(
            models.Enquiry.objects.permitted(self.request.user).only(
                'status', 'id'
            ),
            pk=self.kwargs['enquiry_id'],
        )
        self.check_object_permissions(request, enquiry)

    def post(self, request, *args, **kwargs):

        data = request.data

        if len(json_dumps(data)) > settings.FORM_STORAGE_MAX_SIZE:
            raise ValidationError(_('FORM_STORAGE::SIZE_EXCEEDS'))

        enquiry = self.enquiry

        if enquiry.draft_id:
            models.FormData.objects.filter(id=enquiry.draft_id).update(
                data=request.data
            )

        else:
            draft = models.FormData.objects.create(
                author=request.user, data=request.data
            )

            enquiry.draft_id = draft.id
            enquiry.save(update_fields=('draft_id',))

        return Response({'result': 'ok'})

    def delete(self, request, *args, **kwargs):

        enquiry = self.enquiry

        if enquiry.draft_id:

            draft_id = enquiry.draft_id

            enquiry.draft_id = None
            enquiry.save(update_fields=('draft_id',))

            models.FormData.objects.filter(id=draft_id).delete()

        return Response({'result': 'ok'})
