# coding: utf-8
from django.db.models import F
from rest_framework.response import Response

from procu.api import models
from procu.api.enums import QR, QS
from procu.api.utils import dict_diff, json_dumps, strtobool
from procu.rest import generics
from procu.rest.permissions import ListPermission
from .permissions import QuoteBulkPermission
from .serializers import BulkStatusChange
from ..serializers import Log


def status_repr(status, reason):
    output = f'{QS.i18n[status]}'

    if reason != QR.NONE:
        output += f' ({QR.i18n[reason]})'

    return output


class QuoteBulk(generics.ListCreateAPIView):
    permission_classes = (ListPermission, QuoteBulkPermission)
    serializer_class = BulkStatusChange

    def get_queryset(self):
        return (
            models.Quote.objects.permitted(self.request.user)
            .filter(request__enquiry=self.kwargs['enquiry_id'])
            .select_related('supplier')
            .order_by('status')
        )

    def post_initial(self, request):
        enquiry = self.enquiry = generics.get_object_or_404(
            models.Enquiry.objects.permitted(self.request.user).only(
                'status', 'id'
            ),
            pk=self.kwargs['enquiry_id'],
        )
        self.check_object_permissions(request, enquiry)

    def get(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())

        data = queryset.values('id', title=F('supplier__title'))

        return Response(data)

    def post(self, request, *args, **kwargs):

        user = request.user
        dry_run = strtobool(request.GET.get('dry_run'))

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data

        status = data['status']
        reason = data.get('reason', QR.NONE)

        quotes = models.Quote.objects.select_related('supplier').filter(
            request__enquiry_id=self.kwargs['enquiry_id'], id__in=data['quotes']
        )

        valid = []

        for quote in quotes:

            new_status = status
            new_reason = reason

            if new_status not in quote.available_statuses(user):
                continue

            if new_status == QS.CLOSED:

                if new_reason == QR.NONE:
                    if quote.has_won:
                        new_reason = QR.DELIVERED
                    elif quote.has_offer:
                        new_reason = QR.DECLINED
                    else:
                        new_reason = QR.IGNORED

                if new_reason not in quote.available_reasons(user):
                    continue

            valid.append((quote, new_status, new_reason))

        if dry_run:
            data = [
                {
                    'title': quote.supplier.title,
                    'old': status_repr(quote.status, quote.reason),
                    'new': status_repr(status, reason),
                }
                for quote, status, reason in valid
            ]
            data.sort(key=lambda x: x['title'])

            return Response({'results': data})

        else:
            for quote, status, reason in valid:

                old = Log(quote).data

                quote.status = status
                quote.reason = reason
                quote.save(update_fields=('status', 'reason'))

                new = Log(quote).data

                diff = dict_diff(old, new, show_unchanged=False)

                if diff:
                    models.Log.objects.create(
                        quote=quote,
                        enquiry_id=self.kwargs['enquiry_id'],
                        user=user,
                        type='update_quote',
                        data=json_dumps(diff),
                        old=json_dumps(old),
                        new=json_dumps(new),
                    )

            return Response({'results': 'ok'})
