# coding: utf-8

from django.utils.encoding import force_text
from django.utils.translation import gettext_lazy as _

from procu.api.enums import QR, QS
from procu.jsonschema import schema as js
from procu.rest import serializers


class QStatusField(serializers.Field):
    default_error_messages = {
        'invalid_status': _('FIELD_QSTATUS::INVALID{given}{options}')
    }

    def __init__(self, **kwargs):
        # noinspection PyTypeChecker
        self.name_only = kwargs.pop('name_only', False)
        super().__init__(**kwargs)

    def to_representation(self, value):

        if self.name_only:
            return force_text(QS.i18n[value])

        return {'key': QS.keys[value], 'name': QS.i18n[value]}

    def to_internal_value(self, value):

        for code, name in QS.keys.items():
            if name == value:
                return code

        self.fail(
            'invalid_status', given=value, options=', '.join(QS.keys.values())
        )

    def get_schema(self, write=False, *args, **kwargs):
        if write and self.context['request'].method == 'PATCH':

            quote = self.context['view'].object
            statuses = quote.available_statuses(self.context['request'].user)

            display_names = {QS.keys[s]: QS.i18n[s] for s in statuses}

            return js.String(
                enum=[QS.keys[s] for s in statuses],
                **{'x-names': display_names},
            )

        if self.name_only:
            return js.String()

        return js.Object(('key', js.String()), ('name', js.String()))


class QReasonField(serializers.Field):
    default_error_messages = {
        'invalid_reason': _('FIELD_QREASON::INVALID{given}{options}')
    }

    def __init__(self, **kwargs):
        # noinspection PyTypeChecker
        self.name_only = kwargs.pop('name_only', False)
        super().__init__(**kwargs)

    def to_representation(self, value):

        if self.name_only:
            return force_text(QR.i18n[value])

        return {'key': QR.keys[value], 'name': QR.i18n[value]}

    def to_internal_value(self, value):

        for code, name in QR.keys.items():
            if name == value:
                return code

        self.fail(
            'invalid_reason', given=value, options=', '.join(QR.keys.values())
        )

    def get_schema(self, write=False, *args, **kwargs):

        if write and self.context['request'].method == 'PATCH':

            quote = self.context['view'].object
            reasons = quote.available_reasons(self.context['request'].user)

            display_names = {QR.keys[r]: QR.i18n[r] for r in reasons}

            return js.String(
                enum=[QR.keys[r] for r in reasons], **{'x-names': display_names}
            )

        if self.name_only:
            return js.String()

        return js.Object(('key', js.String()), ('name', js.String()))


class DeadlineAt(serializers.DateTimeField):
    def get_schema(self, *args, **kwargs):
        schema = super().get_schema(*args, **kwargs)
        rfx = self.context['view'].object

        if kwargs.get('write', False):
            schema['default'] = rfx.deadline_at

        return schema
