# coding: utf-8

import logging

from django.conf import settings
from django.template import loader
from rest_framework.response import Response
from startrek_client.exceptions import StartrekServerError

from procu.api import models
from procu.api.enums import LINK
from procu.api.tasks import sync_remotes
from procu.api.utils import get_tracker_client, get_unique_id, parse_terms
from procu.rest import exceptions, generics
from procu.rest.metadata.mixins import CreateMetaMixin
from .permissions import SCSPermission
from .serializers import Create, get_context

logger = logging.getLogger(__name__)


def get_body(context):
    body = loader.render_to_string('scs.txt', context={'data': context})
    return '\n'.join(filter(None, body.split('\n')))


class SCSView(CreateMetaMixin, generics.RetrieveAPIView):
    lookup_url_kwarg = 'quote_id'
    serializer_class = Create
    permission_classes = (SCSPermission,)

    def get_queryset(self):
        return models.Quote.objects.permitted(self.request.user).filter(
            request__enquiry_id=self.kwargs['enquiry_id']
        )

    def retrieve(self, request, *args, **kwargs):
        enquiry_id = self.kwargs['enquiry_id']
        quote_id = self.kwargs['quote_id']
        context = get_context(request.user, enquiry_id, quote_id)
        return Response(context)

    def post(self, request, *args, **kwargs):

        quote = self.object
        supplier = quote.supplier
        rfx = quote.request

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data

        client = get_tracker_client()

        context = {
            'supplier': {'title': supplier.title},
            'legal_entity': data['legal_entity'].title,
            'customer': data['customer'].username,
            'manager': data['manager'].username,
            'reference': data['reference'],
            'subject': data['subject'],
            'delivery_at': data['delivery_at'],
            'terms': data['terms'],
            'url': settings.FRONT_INTERNAL_PREFIX,
            'enquiry_key': rfx.key,
        }

        # OEBS data
        for field in ('cfo', 'task', 'project', 'service'):
            if data.get(field):
                context[field] = f'{data[field].key}: {data[field].name}'

        # ----------------------------------------------------------------------

        if supplier.legal_name and supplier.legal_name != supplier.title:
            context['supplier']['legal_name'] = supplier.legal_name

        subject = (
            f'{"СРОЧНО! " * bool(data["is_urgent"])}'
            f'Завести ЗП: {supplier.legal_name}'
        )

        issue_kwargs = {
            'queue': data['queue'],
            'type': {'name': 'Task'},
            'summary': subject,
            'description': get_body(context),
            'access': [settings.ROBOT_LOGIN],
            'followers': [settings.ROBOT_LOGIN],
            'tags': ['procu'],
            'createdBy': self.request.user.username,
            'attachments': [a.file for a in data['attachments']],
        }

        # ----------------------------------------------------------------------

        quote.terms = parse_terms(data['terms'])
        quote.save(update_fields=['terms'])

        # ----------------------------------------------------------------------

        keys = []

        # SCS
        try:
            issue_kwargs['unique'] = get_unique_id(issue_kwargs)

            scs_issue = client.issues.create(**issue_kwargs)
            keys.append(scs_issue.key)

        except StartrekServerError as exc:
            errors = exc.error_messages

            if exc.errors:
                errors.extend(f'{f}: {v}' for f, v in exc.errors.items())

            if not errors:
                errors.append('Could not create ticket')

            raise exceptions.ValidationError({'detail': errors})

        except Exception:
            logger.exception('Could not create ticket in SCS')
            raise exceptions.ValidationError(
                {'detail': 'Could not create ticket'}
            )

        # LOGIC
        if data['request_np']:
            for a in data['attachments']:
                a.file.seek(0)

            try:
                context['reference'].append(scs_issue.key)

                issue_kwargs['queue'] = settings.LDM_QUEUE
                issue_kwargs['summary'] = 'Уточнить НП'
                issue_kwargs['description'] = get_body(context)
                issue_kwargs['unique'] = get_unique_id(issue_kwargs)
                issue_kwargs['assignee'] = 'alexandrag'
                issue_kwargs['components'] = ['LocalData']
                issue_kwargs['logisticsWorktype'] = 'Прочее'

                if settings.IS_PRODUCTION:
                    issue_kwargs['typeOfEquipment'] = 'Local Data'

                logic_issue = client.issues.create(**issue_kwargs)
                keys.append(logic_issue.key)

            except Exception:
                logger.exception('Could not create ticket in LOGIC')

        sync_remotes.apply_async(
            kwargs={
                'enquiry_id': rfx.enquiry_id,
                'link_to': keys,
                'unlink_from': [],
            }
        )

        # ----------------------------------------------------------------------

        for key in keys:
            models.Link.objects.get_or_create(
                enquiry_id=rfx.enquiry_id,
                key=key,
                defaults={'type': LINK.TRACKER, 'author': request.user},
            )

        # ----------------------------------------------------------------------

        models.SCSTicket.objects.get_or_create(
            ticket=scs_issue.key, quote=quote
        )

        return Response({'key': scs_issue.key, 'summary': subject})
