# coding: utf-8

from itertools import chain

from django.db.models import Prefetch

from procu.api import models
from procu.api.currency.serializers import CurrencyBriefSerializer
from procu.api.utils.decimal import money
from procu.rest import generics
from .permissions import QuoteSummaryPermission
from .serializers import EnquiryProductSerializer, QueryParamsSerializer


class QuoteSummaryView(generics.ListAPIView):
    query_params_serializer_class = QueryParamsSerializer
    permission_classes = (QuoteSummaryPermission,)
    serializer_class = EnquiryProductSerializer

    def post_initial(self, request):
        quote = self.quote = generics.get_object_or_404(
            models.Quote.objects.permitted(self.request.user),
            id=self.kwargs['quote_id'],
            request__enquiry=self.kwargs['enquiry_id'],
        )
        self.check_object_permissions(request, quote)

    def get_queryset(self):
        query_params = self.query_params

        if self.quote.has_offer:
            qp_qs = (
                self.quote.products.prepared(**query_params)
                .filter(quote=self.quote)
                .select_related('currency')
                .order_by('is_replacement', 'pk')
            )
        else:
            qp_qs = models.QuoteProduct.objects.none()

        offers_qs = qp_qs.filter(is_replacement=False)
        replacements_qs = qp_qs.filter(is_replacement=True)

        return models.EnquiryProduct.objects.filter(
            request__enquiry=self.kwargs['enquiry_id']
        ).prefetch_related(
            Prefetch('quote_products', offers_qs, 'offers'),
            Prefetch('quote_products', replacements_qs, 'replacements'),
        )

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        objs = self.paginate_queryset(queryset)

        serializer = self.get_serializer(objs, many=True)

        # Compute total cost

        currency = serializer.context.get('currency')

        curr_set = set()
        total = None

        offers = list(
            chain.from_iterable(
                chain(product.offers, product.replacements) for product in objs
            )
        )

        can_see_prices = serializer.context.get('can_see_prices', False)
        non_empty_offer = any(offer.total is not None for offer in offers)

        if can_see_prices and non_empty_offer and currency:

            total = sum(
                (x.total for x in offers if x.total is not None),
                money(0., ndigits=2),
            )

            if not currency:

                curr_set = curr_set.union(
                    x.currency for x in offers if x.currency is not None
                )

                if len(curr_set) == 1:
                    currency = CurrencyBriefSerializer(curr_set.pop()).data
                else:
                    total = None

        # ---------------------------------------------------

        return self.get_paginated_response(
            {'products': serializer.data, 'currency': currency, 'total': total}
        )

    def get_serializer_context(self):
        user = self.request.user
        enquiry = self.quote.request.enquiry

        context = super().get_serializer_context()

        summary_perm = 'api.summary_enquiry'
        context['can_see_prices'] = summary_perm in enquiry.permissions(user)

        if self.query_params['currency'] is not None:
            context['currency'] = CurrencyBriefSerializer(
                self.query_params['currency']
            ).data

        return context
