# coding: utf-8

from django.utils.translation import gettext_lazy as _

from django.db.models import F

from procu.api import models
from procu.api.common.fields import AttachmentField, TicketsField
from procu.api.enums import LINK
from procu.api.legal_entity.serializers import LegalEntityBrief
from procu.rest import serializers


class Address(serializers.ModelSerializer):
    class Meta:
        model = models.Address
        fields = ('id', 'text', 'label', 'tracker_location')


class Create(serializers.Serializer):

    legal_entity = serializers.PKPrettyField(
        serializer=LegalEntityBrief,
        queryset=models.LegalEntity.objects.filter(is_deleted=False),
        label=_('WAREHOUSE::LEGAL_ENTITY_TO'),
        allow_null=False,
    )
    address = serializers.PKPrettyField(
        serializer=Address,
        queryset=models.Address.objects.filter(is_deleted=False),
        label=_('WAREHOUSE::ADDRESS_TO'),
        allow_null=False,
    )
    reference = TicketsField(label=_('WAREHOUSE::REFERENCE'))
    delivery_at = serializers.DateField(
        label=_('WAREHOUSE::DELIVERY_AT'), allow_null=False
    )
    delivery_orders = serializers.ListSerializer(
        child=serializers.CharField(allow_blank=False),
        allow_empty=False,
        required=False,
        label=_('WAREHOUSE::DELIVERY_ORDERS'),
    )
    subject = serializers.CharField(label=_('WAREHOUSE::SUBJECT'))
    comment = serializers.CharField(
        required=False, allow_blank=True, label=_('WAREHOUSE::COMMENT')
    )
    followers = serializers.ListField(
        child=serializers.CharField(),
        allow_empty=True,
        required=True,
        label=_('WAREHOUSE::FOLLOWERS'),
    )
    attachments = AttachmentField(
        many=True,
        label=_('WAREHOUSE::ATTACHMENTS'),
        allow_empty=True,
        required=False,
    )


def get_context(user, enquiry_id, quote_id):

    quote = (
        models.Quote.objects.permitted(user)
        .values(
            'terms',
            'delivery_at',
            'delivery_orders',
            legal_entity=F('request__legal_entity'),
            address=F('request__address'),
            subject=F('request__subject'),
        )
        .get(request__enquiry_id=enquiry_id, id=quote_id)
    )

    links = list(
        models.Link.objects.filter(
            object__type=LINK.TRACKER, enquiry=enquiry_id
        ).values_list('key', flat=True)
    )

    attachments = list(
        models.Attachment.objects.values('id', 'filename', 'created_at').filter(
            comment__quote_id=quote_id
        )
    )

    # --------------------------------------------------------------------------

    output = {
        'subject': quote['subject'],
        'reference': links,
        'delivery_orders': quote['delivery_orders'],
        'delivery_at': quote['delivery_at'],
        'legal_entity': (
            {'id': quote['legal_entity']} if quote['legal_entity'] else None
        ),
        'address': {'id': quote['address']} if quote['address'] else None,
        'attachments': attachments,
        'comment': '',
    }

    return output
