# coding: utf-8

from itertools import chain

from django.core import mail
from django.http import Http404
from django.utils.encoding import force_text
from rest_framework.response import Response

from procu.api import models
from procu.api.enquiry.quote.notify import quote_notify
from procu.api.enums import ER, ES, QR, QS
from procu.api.push.external import (
    notify_enquiry_changed,
    notify_quote_address_changed,
)
from procu.api.utils import dict_diff, json_dumps
from procu.rest import generics
from . import permissions, serializers
from ..publish.serializers import Log


class RequestRetrieveUpdate(generics.RetrieveUpdateAPIView):
    serializer_class = serializers.Request
    output_serializer_class = serializers.Request
    permission_classes = (permissions.RequestUpdatePermission,)
    lookup_url_kwarg = 'enquiry_id'
    lookup_field = 'enquiry_id'

    def get_queryset(self):
        return (
            models.Request.objects.permitted(self.request.user)
            .select_related('address', 'legal_entity')
            .prefetch_related('attachments', 'products')
        )

    def get(self, request, *args, **kwargs):
        try:
            return super().get(request, *args, **kwargs)
        except Http404:
            return Response({})

    def perform_update(self, serializer):

        user = self.request.user
        old = Log(serializer.instance).data
        old_products = list(
            models.EnquiryProduct.objects.values(
                'id', 'name', 'qty', 'comment'
            ).filter(request_id=serializer.instance.id)
        )

        # ----------------------------------------------------------------------

        # Actual update
        instance = super().perform_update(serializer)

        # ----------------------------------------------------------------------
        # Logging: request

        new = Log(instance).data

        highlights = {}
        diff = dict_diff(old, new, show_unchanged=False)

        if diff:
            for field in (
                'address',
                'legal_entity',
                'no_replacement',
                'description',
                'subject',
            ):
                if field in diff:
                    highlights[field] = True

            old_attachments = {a['id'] for a in old['attachments']}
            new_attachments = {a['id'] for a in new['attachments']}

            if new_attachments - old_attachments:
                highlights['attachments'] = new_attachments - old_attachments

            models.Log.objects.create(
                enquiry_id=self.kwargs['enquiry_id'],
                user=user,
                type='update_request',
                data=json_dumps(diff),
            )

        # ----------------------------------------------------
        # Products

        new_products = list(
            models.EnquiryProduct.objects.values(
                'id', 'name', 'qty', 'comment'
            ).filter(request_id=instance.id)
        )

        product_diff = []
        products_hl = {}

        product_fields = ('name', 'qty', 'comment')
        dummy_product = {f: '' for f in product_fields}

        product_map = {product['id']: product for product in old_products}

        for item in new_products:

            item_id = item.get('id')

            products_hl[item_id] = hl_fields = {}

            if item_id not in product_map:
                product_diff.append(
                    dict_diff(dummy_product, item, fields=product_fields)
                )
                hl_fields.update({f: True for f in product_fields})

            else:
                product = product_map.pop(item_id)
                diff = dict_diff(product, item, fields=product_fields)
                if diff:
                    product_diff.append(diff)

                    for f in product_fields:
                        if (
                            isinstance(diff.get(f, None), dict)
                            and '__old__' in diff[f]
                        ):
                            hl_fields[f] = True

        for product in product_map.values():
            product_diff.append(
                dict_diff(product, dummy_product, fields=product_fields)
            )

        if product_diff:
            highlights['products'] = products_hl

            models.Log.objects.create(
                quote=None,
                enquiry_id=self.kwargs['enquiry_id'],
                user=self.request.user,
                type='replace_request_products',
                data=json_dumps(product_diff),
            )

        if highlights:

            # ------------------------------------------------------------------
            # General notification about enquiry changes

            if instance.status < ES.SHIPPED:
                quote_ids = instance.quotes.filter(
                    status__gt=QS.DRAFT, status__lt=QS.CLOSED
                ).values_list('id', flat=True)

                messages = chain.from_iterable(
                    notify_enquiry_changed(id, self.request.user, highlights)
                    for id in quote_ids
                )

                with mail.get_connection(fail_silently=True) as conn:
                    conn.send_messages(messages)

            # ------------------------------------------------------------------
            # Special notification for those in status SHIPPED

            if 'address' in highlights:
                quote_ids = instance.quotes.filter(
                    status=QS.SHIPPED
                ).values_list('id', flat=True)

                messages = chain.from_iterable(
                    notify_quote_address_changed(id, self.request.user)
                    for id in quote_ids
                )

                with mail.get_connection(fail_silently=True) as conn:
                    conn.send_messages(messages)

        return instance


# ------------------------------------------------------------------------------


def get_quote_context(obj):

    return {
        'status': force_text(QS.i18n[obj['status']]),
        'reason': force_text(QR.i18n[obj['reason']]),
        'deadline_at': obj['deadline_at'],
    }


class RequestUpdateDeadline(generics.RetrieveUpdateAPIView):
    serializer_class = serializers.RetrieveUpdateDeadline
    permission_classes = (permissions.RequestUpdatePermission,)
    lookup_url_kwarg = 'enquiry_id'
    lookup_field = 'enquiry_id'

    def get_queryset(self):
        return models.Request.objects.permitted(self.request.user)

    def update(self, request, *args, **kwargs):

        rfx = self.object
        user = self.request.user

        serializer = self.get_serializer(rfx, data=request.data, partial=True)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data

        # ----------------------------------------------------------------------

        rfx_old = {
            'status': force_text(ES.i18n[rfx.status]),
            'reason': force_text(ER.i18n[rfx.reason]),
            'deadline_at': (
                rfx.deadline_at.isoformat() if rfx.deadline_at else None
            ),
        }

        olds = list(
            models.Quote.objects.values('id', 'status', 'reason', 'deadline_at')
            .filter(request=rfx)
            .order_by('id')
        )

        # ----------------------------------------------------------------------

        models.Quote.objects.filter(request=rfx).update(
            deadline_at=data['deadline_at']
        )

        # ----------------------------------------------------------------------
        # Timeline: request

        rfx.refresh_from_db()

        rfx_new = {
            'status': force_text(ES.i18n[rfx.status]),
            'reason': force_text(ER.i18n[rfx.reason]),
            'deadline_at': (
                rfx.deadline_at.isoformat() if rfx.deadline_at else None
            ),
        }

        diff = dict_diff(rfx_old, rfx_new, show_unchanged=False)

        if diff:
            models.Log.objects.create(
                enquiry_id=self.kwargs['enquiry_id'],
                user=self.request.user,
                type='update_request',
                data=json_dumps(diff),
            )

        news = list(
            models.Quote.objects.values('id', 'status', 'reason', 'deadline_at')
            .filter(request=rfx)
            .order_by('id')
        )

        # ----------------------------------------------------------------------
        # Notifications

        messages = []

        for old_quote, new_quote in zip(olds, news):

            quote_id = new_quote['id']

            messages.extend(
                quote_notify(
                    user,
                    quote_id,
                    (old_quote['status'], new_quote['status']),
                    get_quote_context(old_quote),
                    get_quote_context(new_quote),
                )
            )

        with mail.get_connection(fail_silently=True) as conn:
            conn.send_messages(messages)

        # ----------------------------------------------------------------------

        return Response(serializer.validated_data)
