# coding: utf-8

from django.template.loader import get_template
from django.utils.translation import gettext_lazy as _

from procu.api import models
from procu.api.common.fields import EmployeeField
from procu.api.enquiry.fields import Fields, OracleFields
from procu.api.enums import LINK
from procu.api.tasks import sync_remotes
from procu.rest import serializers

EXISTING_ONLY = {'is_deleted': False}

SERVICES_TEMPLATE = get_template('services.txt')


# ------------------------------------------------------------------------------


class Enquiry(serializers.ModelSerializer):
    check_for_nested_writes = False

    description = Fields.description()
    products = Fields.products()
    legal_entity = Fields.legal_entity()
    address = Fields.address()
    attachments = Fields.attachments()
    due_at = Fields.due_at()

    cfo = OracleFields.cfo()
    project = OracleFields.project()
    task = OracleFields.task()
    mvp = OracleFields.mvp()
    service = OracleFields.service()
    budget_line = OracleFields.budget_line()
    system = OracleFields.system()
    subsystem = OracleFields.subsystem()

    class Meta:
        model = models.Enquiry
        fields = (
            # From model
            'id',
            'no_replacement',
            # From serializer
            'description',
            'products',
            'legal_entity',
            'address',
            'attachments',
            'due_at',
            'cfo',
            'project',
            'task',
            'mvp',
            'service',
            'budget_line',
            'system',
            'subsystem',
        )

    def update(self, instance, data):

        if 'attachments' in data:

            data['attachments'] = [
                att
                for att in data['attachments']
                if att.enquiry is None or att.enquiry.id == instance.id
            ]

        instance = super().update(instance, data)

        return instance


class Header(serializers.ModelSerializer):

    note = serializers.CharField(read_only=True)
    assignee = Fields.assignee()
    category = Fields.category()
    priority = Fields.priority()
    author = Fields.author(read_only=True)
    deadline_at = Fields.deadline_at(read_only=True)

    status = Fields.status()
    reason = Fields.reason()
    state = Fields.state()

    class Meta:
        model = models.Enquiry
        fields = (
            'id',
            'key',
            'subject',
            'note',
            'status',
            'reason',
            'assignee',
            'author',
            'deadline_at',
            'created_at',
            'category',
            'priority',
            'state',
        )

        read_only_fields = ('key',)


# ------------------------------------------------------------------------------


class CreateMixin(object):
    check_for_nested_writes = False

    def create(self, data):

        user = data['author']
        tickets = data.pop('reference', [])

        data['attachments'] = [
            att for att in data.get('attachments', ()) if att.enquiry is None
        ]

        instance = super().create(data)

        # ---------------------------------------------------
        # Create links

        if tickets:
            models.Link.objects.bulk_create(
                [
                    models.Link(
                        author=user,
                        enquiry=instance,
                        type=LINK.TRACKER,
                        key=key,
                    )
                    for key in tickets
                ]
            )

            sync_remotes.apply_async(
                kwargs={
                    'enquiry_id': instance.id,
                    'link_to': tickets,
                    'unlink_from': [],
                }
            )

        return instance


class Create(CreateMixin, serializers.ModelSerializer):
    reference = Fields.reference()
    attachments = Fields.attachments()
    assignee = Fields.assignee(filter=EXISTING_ONLY)
    legal_entity = Fields.legal_entity(filter=EXISTING_ONLY)
    address = Fields.address(filter=EXISTING_ONLY)
    category = Fields.category(filter=EXISTING_ONLY)
    priority = Fields.priority()
    products = Fields.products()
    description = Fields.description()
    due_at = Fields.due_at()

    cfo = OracleFields.cfo()
    project = OracleFields.project()
    task = OracleFields.task()
    mvp = OracleFields.mvp()
    service = OracleFields.service()
    budget_line = OracleFields.budget_line()
    system = OracleFields.system()
    subsystem = OracleFields.subsystem()

    class Meta:
        model = models.Enquiry
        fields = (
            # From model
            'id',
            'subject',
            'no_replacement',
            # From serializer
            'due_at',
            'reference',
            'attachments',
            'assignee',
            'legal_entity',
            'address',
            'category',
            'priority',
            'products',
            'description',
            'cfo',
            'project',
            'task',
            'mvp',
            'service',
            'budget_line',
            'system',
            'subsystem',
        )


class CreateForCustomer(CreateMixin, serializers.ModelSerializer):
    reference = Fields.reference()
    attachments = Fields.attachments()
    products = Fields.products()
    description = Fields.description()
    due_at = Fields.due_at()

    cfo = OracleFields.cfo()
    project = OracleFields.project()
    task = OracleFields.task()
    mvp = OracleFields.mvp()
    service = OracleFields.service()
    budget_line = OracleFields.budget_line()
    system = OracleFields.system()
    subsystem = OracleFields.subsystem()

    class Meta:
        model = models.Enquiry
        fields = (
            # From model
            'id',
            'subject',
            'no_replacement',
            # From serializer
            'description',
            'due_at',
            'reference',
            'attachments',
            'products',
            'cfo',
            'project',
            'task',
            'mvp',
            'service',
            'budget_line',
            'system',
            'subsystem',
        )


class CreateForServices(CreateMixin, serializers.ModelSerializer):

    subject = serializers.CharField(
        label=_('SERVICE_PROCUREMENT::TARGET'),
        max_length=255,
        allow_blank=False,
        style={
            'x-hint': 'Что планируем закупить, '
            'или конечный результат оказания услуг'
        },
    )

    tasks = serializers.CharField(
        label=_('SERVICE_PROCUREMENT::TASKS'),
        allow_blank=False,
        style={'x-hint': 'Конкретизируйте цель'},
    )
    description = serializers.CharField(
        label=_('SERVICE_PROCUREMENT::DESCRIPTION'),
        allow_blank=False,
        style={
            'x-hint': (
                'Опишите характеристики, свойства, '
                'качество, ссылка на продукт'
            )
        },
    )
    volume = serializers.CharField(
        label=_('SERVICE_PROCUREMENT::VOLUME'),
        allow_blank=False,
        style={
            'x-hint': 'Количество товара или прогнозируемый объем на проект'
        },
    )

    criteria = serializers.ListSerializer(
        child=serializers.CharField(allow_blank=False, max_length=255),
        allow_empty=True,
        label=_('SERVICE_PROCUREMENT::CRITERIA'),
    )

    due_at = Fields.due_at(
        label=_('SERVICE_PROCUREMENT::DUE_AT'),
        style={
            'x-hint': 'Не позднее какой даты должен быть поставлен товар '
            'или оказаны услуги'
        },
    )
    budget = serializers.CharField(
        label=_('SERVICE_PROCUREMENT::BUDGET'),
        max_length=255,
        style={
            'x-hint': 'Сколько денег заложено в бюджете на эту закупку '
            '(без НДС, указать валюту)'
        },
    )
    budget_owner = EmployeeField(label=_('SERVICE_PROCUREMENT::BUDGET_OWNER'))
    info = serializers.CharField(
        label=_('SERVICE_PROCUREMENT::INFO'),
        allow_blank=True,
        style={'x-hint': 'Всё, что вы считаете важным'},
    )

    cfo = OracleFields.cfo()
    project = OracleFields.project()
    task = OracleFields.task()
    mvp = OracleFields.mvp()
    service = OracleFields.service()
    budget_line = OracleFields.budget_line()
    system = OracleFields.system()
    subsystem = OracleFields.subsystem()

    attachments = Fields.attachments()

    class Meta:
        model = models.Enquiry
        fields = (
            'subject',
            'due_at',
            'cfo',
            'project',
            'task',
            'mvp',
            'service',
            'budget_line',
            'system',
            'subsystem',
            'attachments',
            'tasks',
            'description',
            'volume',
            'criteria',
            'budget',
            'budget_owner',
            'info',
        )

    def validate(self, data):

        internal_comment = SERVICES_TEMPLATE.render(context={'data': data})
        internal_comment = '\n'.join(filter(None, internal_comment.split('\n')))
        data['internal_comment'] = internal_comment

        for f in (
            'tasks',
            'description',
            'volume',
            'criteria',
            'budget',
            'budget_owner',
            'info',
        ):
            data.pop(f)

        return data


# ------------------------------------------------------------------------------


class Log(serializers.Serializer):

    reference = serializers.SerializerMethodField()
    author = Fields.author()
    manager = Fields.manager()
    legal_entity = Fields.legal_entity()
    address = Fields.address()
    category = Fields.category()
    attachments = Fields.attachments()
    due_at = Fields.due_at(format=None)
    status = Fields.status(name_only=True)
    reason = Fields.reason(name_only=True)
    priority = Fields.priority(name_only=True)

    no_replacement = serializers.BooleanField()
    internal_comment = serializers.CharField()
    subject = serializers.CharField()

    responsibility_center = serializers.SerializerMethodField()
    program = serializers.SerializerMethodField()
    project = serializers.SerializerMethodField()
    task = serializers.SerializerMethodField()
    location = serializers.SerializerMethodField()
    service = serializers.SerializerMethodField()
    budget_line = serializers.SerializerMethodField()
    system = serializers.SerializerMethodField()
    subsystem = serializers.SerializerMethodField()

    @staticmethod
    def get_reference(data):
        links = list(
            models.Link.objects.filter(
                object__type=LINK.TRACKER, enquiry=data.id
            ).values_list('key', flat=True)
        )
        return links

    @staticmethod
    def get_responsibility_center(data):
        if data.cfo:
            return f'{data.cfo.key}: {data.cfo.name}'
        return ''

    @staticmethod
    def get_program(data):
        if data.program:
            return f'{data.program.key}: {data.program.name}'
        return ''

    @staticmethod
    def get_project(data):
        if data.project:
            return f'{data.project.key}: {data.project.name}'
        return ''

    @staticmethod
    def get_task(data):
        if data.task:
            return f'{data.task.key}: {data.task.name}'
        return ''

    @staticmethod
    def get_location(data):
        if data.mvp:
            return f'{data.mvp.key}: {data.mvp.name}'
        return ''

    @staticmethod
    def get_service(data):
        if data.service:
            return f'{data.service.key}: {data.service.name}'
        return ''

    @staticmethod
    def get_budget_line(data):
        if data.budget_line:
            return f'{data.budget_line.key}: {data.budget_line.name}'
        return ''

    @staticmethod
    def get_system(data):
        if data.system:
            return f'{data.system.key}: {data.system.name}'
        return ''

    @staticmethod
    def get_subsystem(data):
        if data.subsystem:
            return f'{data.subsystem.key}: {data.subsystem.name}'
        return ''

    class Meta:
        fields = (
            # From model
            'id',
            'subject',
            'created_at',
            'updated_at',
            # From serializer
            'due_at',
            'reference',
            'author',
            'manager',
            'legal_entity',
            'address',
            'category',
            'attachments',
            'status',
            'reason',
            'priority',
            'responsibility_center',
            'program',
            'project',
            'task',
            'location',
            'service',
            'budget_line',
            'system',
            'subsystem',
            'no_replacement',
            'internal_comment',
        )
