# coding: utf-8

from collections import defaultdict
from decimal import Decimal
from io import BytesIO

import xlsxwriter
from django.utils.translation import gettext as _
from xlsxwriter.utility import xl_rowcol_to_cell as cell


class ExcelController(object):
    def __init__(self, config, suppliers, rows):

        self.config = config
        self.suppliers = suppliers
        self.rows = rows

        self.output = BytesIO()
        self.workbook = wb = xlsxwriter.Workbook(
            self.output, {'in_memory': True}
        )

        # ----------------------------------------------------------------------
        # Format attributes

        title = {'align': 'vjustify', 'text_wrap': True}

        highlighted = {
            'bg_color': '#ffffb3',
            'border': 1,
            'border_color': '#000000',
        }

        hl_snapshot = {
            'bg_color': '#eef2fb',
            'border': 1,
            'border_color': '#000000',
        }

        center = {'align': 'center'}

        right = {'align': 'right'}

        multiplier = {'num_format': '× 0', 'align': 'left'}

        currency = self.config['currency']
        prefix = '[$%s] ' % currency.prefix if currency.prefix else ''
        suffix = ' [$%s]' % currency.suffix if currency.suffix else ''

        money_exact = {'num_format': '{}# ##0.00{}'.format(prefix, suffix)}

        money_round = {'num_format': '{}# ##0{}'.format(prefix, suffix)}

        # ----------------------------------------------------------------------
        # Workbook formats

        self.f_offer = wb.add_format(title)
        self.f_replacement = wb.add_format({**title, **highlighted})
        self.f_snapshot = wb.add_format({**title, **right, **hl_snapshot})

        self.f_supplier = wb.add_format(center)
        self.f_qty = wb.add_format(multiplier)

        self.f_money = wb.add_format(money_exact)
        self.f_money_round = wb.add_format(money_round)

        if self.config['rounded']:
            self.f_total = wb.add_format({**money_round, **center})
        else:
            self.f_total = wb.add_format({**money_exact, **center})

        # ----------------------------------------------------------------------

    def format_prices(self, sheet, first_cell, last_cell):

        cell_range = '{}:{}'.format(first_cell, last_cell)

        if self.config['rounded']:
            sheet.conditional_format(
                cell_range,
                {
                    'type': 'formula',
                    'format': self.f_money_round,
                    'criteria': '={c}=TRUNC({c})'.format(c=first_cell),
                },
            )
            sheet.conditional_format(
                cell_range,
                {
                    'type': 'formula',
                    'format': self.f_money,
                    'criteria': '={c}<>TRUNC({c})'.format(c=first_cell),
                },
            )

        else:
            sheet.conditional_format(
                cell_range, {'type': 'no_errors', 'format': self.f_money}
            )

    def add_enquiry_summary(self):

        n_rows = len(self.rows)

        sheet = self.workbook.add_worksheet(_('SUMMARY_EXPORT::SHEET'))
        sheet.freeze_panes(1, 0)

        # Header
        sheet.write_row(
            0, 0, (_('SUMMARY_EXPORT::PRODUCT_NAME'), _('SUMMARY_EXPORT::QTY'))
        )

        # Mapping supplier -> column id
        col_ids = {}

        effective_coefs = {}
        payment_terms = {}

        for i, supplier in enumerate(self.suppliers):

            if self.config['totals']:
                col = 2 + i
                sheet.write(0, col, supplier['title'], self.f_supplier)

            else:
                col = 2 + 2 * i

                # Merge supplier title cells
                sheet.merge_range(
                    0, col, 0, col + 1, supplier['title'], self.f_supplier
                )

                # Merge total price cells
                sheet.merge_range(
                    n_rows + 2, col, n_rows + 2, col + 1, '', self.f_supplier
                )

                # Merge payment terms cells
                sheet.merge_range(
                    n_rows + 3, col, n_rows + 3, col + 1, '', self.f_supplier
                )

                # Merge effective price cells
                sheet.merge_range(
                    n_rows + 4, col, n_rows + 4, col + 1, '', self.f_supplier
                )

            col_ids[supplier['id']] = col

            effective_coefs[supplier['id']] = supplier.get('effective_coef')
            payment_terms[supplier['id']] = supplier['terms'].get('comment', '')

            # Apply conditional price style
            first_cell = cell(1, col)
            last_cell = cell(n_rows, col)

            self.format_prices(sheet, first_cell, last_cell)

        # Supplier column
        sheet.set_column(0, 0, 40)

        # ----------------------------------------------------------------------

        non_snapshot_rows = []

        totals = defaultdict(lambda: Decimal('0.00'))

        for i, row in enumerate(self.rows, 1):

            # Product title
            if row['is_snapshot']:
                fmt = self.f_snapshot
            elif row['is_replacement']:
                fmt = self.f_replacement
                row['name'] = '      ' + row['name']
            else:
                fmt = self.f_offer

            if not row['is_snapshot']:
                non_snapshot_rows.append(i)

            sheet.write(i, 0, row['name'], fmt)

            # Requested quantity
            sheet.write(i, 1, row['qty'] or '')

            for offer in row['offers'].values():

                col = col_ids[offer['supplier']]

                # Price
                if offer['value'] is not None:
                    sheet.write_number(i, col, offer['value_'])

                    if not row['is_snapshot']:
                        qty = 1 if self.config['totals'] else offer['qty']
                        totals[offer['supplier']] += offer['value_'] * qty

                if not self.config['totals']:
                    sheet.write_number(i, col + 1, offer['qty'], self.f_qty)

                # Cell comment
                comments = [
                    '%s: %s' % (_('SUMMARY_EXPORT::QTY'), offer['qty'])
                    if (offer['qty_expected'] != offer['qty'])
                    else '',
                    '%s: %s' % (_('SUMMARY_EXPORT::COMMENT'), offer['comment'])
                    if offer['comment']
                    else '',
                    '%s: %s'
                    % (
                        _('SUMMARY_EXPORT::DELIVERY_TIME'),
                        offer['delivery_time'],
                    )
                    if offer['delivery_time']
                    else '',
                ]
                comment = '\n\n'.join(filter(None, comments))

                if comment:
                    sheet.write_comment(i, col, comment)

        # Totals
        sheet.write(n_rows + 2, 0, _('SUMMARY_EXPORT::FINAL'))

        for supplier, col in col_ids.items():

            if self.config['totals']:
                formula = '+'.join(f'{cell(r, col)}' for r in non_snapshot_rows)
            else:
                formula = '+'.join(
                    f'{cell(r, col)}*{cell(r, col+1)}'
                    for r in non_snapshot_rows
                )

            formula = f'={formula}'

            sheet.write_formula(
                n_rows + 2, col, formula, self.f_total, totals[supplier]
            )

        # Effective totals and payment terms
        sheet.write(n_rows + 3, 0, _('SUMMARY_EXPORT::PAYMENT_TERMS'))
        sheet.write(n_rows + 4, 0, _('SUMMARY_EXPORT::EFFECTIVE'))

        for supplier, col in col_ids.items():

            # Payment terms
            sheet.write(
                n_rows + 3, col, payment_terms[supplier], self.f_supplier
            )

            coef = effective_coefs.get(supplier)

            if coef is None:
                continue

            # Effective totals
            sheet.write_formula(
                n_rows + 4,
                col,
                f'={cell(n_rows+2, col)} * {coef}',
                self.f_total,
                coef * float(totals[supplier]),
            )

    def add_quote_summary(self, supplier):

        # Excel has limitations on worksheet titles
        worksheet_title = supplier['title'][:31]

        sheet = self.workbook.add_worksheet(worksheet_title)
        sheet.freeze_panes(1, 0)

        sheet.write_row(
            0,
            0,
            (
                _('SUMMARY_EXPORT::PRODUCT_NAME'),
                _('SUMMARY_EXPORT::QTY_EXPECTED'),
                _('SUMMARY_EXPORT::QTY_OFFERED'),
                _('SUMMARY_EXPORT::TOTAL')
                if self.config['totals']
                else _('SUMMARY_EXPORT::PRICE'),
                _('SUMMARY_EXPORT::DELIVERY_TIME'),
                _('SUMMARY_EXPORT::COMMENT'),
            ),
        )

        sheet.set_column(0, 0, 40)

        # Apply conditional price style
        first_cell = cell(1, 3)
        last_cell = cell(len(self.rows), 3)
        self.format_prices(sheet, first_cell, last_cell)

        row_id = 1

        for row in self.rows:

            if supplier['id'] not in row['offers']:
                continue

            offer = row['offers'][supplier['id']]

            if offer['is_snapshot']:
                fmt = self.f_snapshot
            elif offer['is_replacement']:
                fmt = self.f_replacement
                offer['name'] = '      ' + offer['name']
            else:
                fmt = self.f_offer

            sheet.write(row_id, 0, row['name'], fmt)
            sheet.write(row_id, 1, offer['qty_expected'])
            sheet.write(row_id, 2, offer['qty'])

            if offer['value_'] is not None:
                sheet.write_number(row_id, 3, offer['value_'])

            sheet.write(row_id, 4, offer['delivery_time'])
            sheet.write(row_id, 5, offer['comment'])

            row_id += 1
