# coding: utf-8

import logging

from django.conf import settings
from django.db.models import Prefetch
from django.utils.translation import gettext_lazy as _
from rest_framework.response import Response
from startrek_client.exceptions import StartrekServerError

from procu.api import models, tasks
from procu.api.enums import LINK
from procu.api.utils import get_tracker_client, get_unique_id, json_dumps
from procu.rest import exceptions, generics
from procu.rest.permissions import EntryPermission
from . import serializers

logger = logging.getLogger(__name__)


class EnquiryCreateTicket(generics.RetrieveAPIView, generics.CreateMetaMixin):
    permission_classes = (EntryPermission,)
    lookup_url_kwarg = 'enquiry_id'

    def get_serializer_class(self):
        if self.request.method == 'GET':
            return serializers.TicketRetrieve
        else:
            return serializers.TicketCreate

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .select_related(
                'cfo',
                'program',
                'project',
                'task',
                'service',
                'mvp',
                'budget_line',
                'system',
                'subsystem',
                'legal_entity',
                'address',
                'author',
            )
            .prefetch_related(
                'attachments',
                Prefetch(
                    'links',
                    models.Link.objects.filter(object__type=LINK.TRACKER),
                ),
            )
        )

    def post(self, request, *args, **kwargs):

        enquiry = self.object

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        data = serializer.validated_data

        attachments = data.pop('attachments')
        location = data.pop('location')

        issue_kwargs = {
            'queue': location['queue'],
            'type': {'key': location['type']},
            'access': [settings.ROBOT_LOGIN],
            'followers': [settings.ROBOT_LOGIN],
            'tags': ['procu'],
            'createdBy': request.user.username,
            'attachments': [a.file for a in attachments],
            **data,
        }

        issue_kwargs['unique'] = get_unique_id(issue_kwargs)

        client = get_tracker_client()

        try:
            issue = client.issues.create(**issue_kwargs)

        except StartrekServerError as exc:
            errors = exc.error_messages

            if exc.errors:
                errors.extend(f'{f}: {v}' for f, v in exc.errors.items())

            if not errors:
                errors.append(_('ENQUIRY::ERROR_COULD_NOT_CREATE'))

            raise exceptions.ValidationError({'detail': errors})

        except Exception:
            logger.exception('Could not create ticket')
            raise exceptions.ValidationError(
                {'detail': _('ENQUIRY::ERROR_COULD_NOT_CREATE')}
            )

        tasks.sync_remotes.apply_async(
            kwargs={
                'enquiry_id': enquiry.id,
                'link_to': [issue.key],
                'unlink_from': [],
            }
        )

        models.Link.objects.create(
            author=request.user,
            enquiry=enquiry,
            type=LINK.TRACKER,
            key=issue.key,
        )

        new = {'keys': [issue.key]}

        models.Log.objects.create(
            quote=None,
            enquiry_id=enquiry.id,
            user=request.user,
            type='create_enquiry_links',
            data=json_dumps(new),
            new=json_dumps(new),
        )

        return Response({'key': issue.key, 'summary': issue_kwargs['summary']})
