# coding: utf-8

from django.core import mail
from django.db.models import F, FilteredRelation, Q, Value
from django.db.models.functions import Concat
from django.http import Http404
from django.utils.encoding import force_text
from django.utils.translation import gettext_lazy as _
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.response import Response

from procu.api import models
from procu.api.enums import ER, ES, PRIORITY, QR, QS
from procu.api.push.internal import (
    notify_enquiry_assigned,
    notify_enquiry_created,
)
from procu.api.user.serializers import UserBrief
from procu.api.utils import dict_diff, is_readonly, json_dumps
from procu.rest import generics, pagination
from procu.rest.permissions import EntryPermission, ListPermission
from . import filters, permissions, serializers
from .quote.notify import quote_notify


class EnquiryList(generics.GenericAPIView):
    pagination_class = pagination.PageNumberPagination
    permission_classes = (ListPermission,)

    filter_backends = (
        DjangoFilterBackend,
        filters.EnquiryOrderingFilter,
        filters.StatusFilter,
        filters.YPSearchFilter,
        filters.PriorityFilter,
        filters.SupplierFilter,
        filters.AddressFilter,
        filters.CategoryFilter,
        filters.AuthorFilter,
        filters.ManagerFilter,
        filters.LegalEntityFilter,
        filters.CFOFilter,
    )

    ordering_fields = (
        'id',
        'updated_at',
        'subject',
        'status',
        'priority',
        'deadline_at',
        'delivery_at',
        'author__last_name',
        'manager__last_name',
    )
    ordering = ('-updated_at',)

    search_fields = ('id', 'subject', 'links__key', 'internal_comment')

    def get(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())

        users = models.User.objects.filter(is_staff=True).values(
            'id',
            'username',
            'email',
            'is_staff',
            'is_deleted',
            full_name=Concat('first_name', Value(' '), 'last_name'),
        )
        users = {u['id']: u for u in users}

        page = self.paginate_queryset(queryset)

        for item in page:
            item['author'] = users.get(item['author'])
            item['manager'] = users.get(item['manager'])

            item['reason'] = {
                'key': ER.keys[item['reason']],
                'name': ER.i18n[item['reason']],
            }

            item['status'] = {
                'key': ES.keys[item['status']],
                'name': ES.i18n[item['status']],
            }

            item['priority'] = {
                'key': PRIORITY.keys[item['priority']],
                'name': PRIORITY.i18n[item['priority']],
            }

        return self.get_paginated_response(page)

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .annotate(
                enquiry_note=FilteredRelation(
                    'notes', condition=Q(notes__user=self.request.user)
                )
            )
            .values(
                'id',
                'subject',
                'author',
                'manager',
                'updated_at',
                'status',
                'reason',
                'priority',
                note=F('enquiry_note__text'),
                deadline_at=F('request__deadline_at'),
                delivery_at=F('request__delivery_at'),
            )
        )


# ------------------------------------------------------------------------------


class EnquiryCreate(generics.CreateAPIView):
    permission_classes = (ListPermission,)
    output_serializer_class = serializers.Enquiry

    def get_queryset(self):
        return models.Enquiry.objects.permitted(self.request.user)

    def get_serializer_class(self):

        form_type = self.request.GET.get('type', 'procu')

        if form_type == 'goods':
            return serializers.CreateForCustomer
        elif form_type == 'services':
            return serializers.CreateForServices
        else:
            return serializers.Create

    def perform_create(self, serializer):

        user = self.request.user

        # Actual creation
        instance = serializer.save(author=user)

        new = serializers.Log(serializer.instance).data

        models.Log.objects.create(
            enquiry=serializer.instance,
            user=user,
            type='create_enquiry',
            new=json_dumps(new),
        )

        # ----------------------------------------------------
        # Products

        product_diff = []

        product_fields = ('name', 'qty', 'comment')
        dummy_product = {f: '' for f in product_fields}

        for item in instance.initial_products:
            product_diff.append(
                dict_diff(dummy_product, item, fields=product_fields)
            )

        if product_diff:
            models.Log.objects.create(
                enquiry_id=instance.id,
                user=user,
                type='replace_enquiry_products_diff',
                data=json_dumps(product_diff),
            )

        models.EnquiryLog.objects.create(
            enquiry_id=instance.id,
            state=instance.state,
            status=instance.status,
            reason=instance.reason,
        )

        # ------------------------------------------------------------------
        # Notifications for a new assignee and equiry creation

        messages = []

        if instance.manager_id not in (None, user.id):
            messages.extend(notify_enquiry_assigned(instance.id, user))

        if instance.manager_id is None:
            messages.extend(notify_enquiry_created(instance.id, user))

        with mail.get_connection(fail_silently=True) as conn:
            conn.send_messages(messages)

        return instance


# ------------------------------------------------------------------------------


class EnquiryPermissionDenied(Exception):
    def __init__(self, data, *args, **kwargs):
        self.data = data


class EnquiryObjectMixin(object):
    def handle_exception(self, exc):
        if isinstance(exc, EnquiryPermissionDenied):
            return Response(exc.data, status=403)

        return super().handle_exception(exc)

    def get_object(self):
        try:
            return super().get_object()
        except Http404 as exc:

            try:
                obj = models.Enquiry.objects.get(id=self.kwargs['enquiry_id'])

                users = filter(None, {obj.author, obj.manager})
                users = filter(
                    lambda x: x.has_perm('api.progress_enquiry'), users
                )

                raise EnquiryPermissionDenied(
                    {
                        'error_code': 'ENQUIRY_PERMISSION_DENIED',
                        'error_title': ' ',
                        'users': UserBrief(users, many=True).data,
                        'detail': _('ERRORS::ENQUIRY_PERMISSION_DENIED'),
                    }
                )

            except models.Enquiry.DoesNotExist:
                raise exc


class EnquiryRetrieveUpdate(EnquiryObjectMixin, generics.RetrieveUpdateAPIView):
    serializer_class = serializers.Enquiry
    permission_classes = (EntryPermission,)
    lookup_url_kwarg = 'enquiry_id'

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .select_related('author', 'legal_entity', 'address')
            .prefetch_related('attachments')
        )

    def perform_update(self, serializer):

        user = self.request.user
        old = serializers.Log(serializer.instance).data
        old_products = serializer.instance.initial_products

        # ----------------------------------------------------------------------

        # Actual update
        instance = super().perform_update(serializer)

        # ----------------------------------------------------------------------

        new = serializers.Log(instance).data
        new_products = instance.initial_products

        diff = dict_diff(old, new, show_unchanged=False)

        if diff:
            models.Log.objects.create(
                enquiry_id=instance.id,
                user=user,
                type='update_enquiry',
                data=json_dumps(diff),
            )

        # ----------------------------------------------------
        # Products

        if old_products != new_products:
            product_diff = []

            product_fields = ('name', 'qty', 'comment')
            dummy_product = {f: '' for f in product_fields}

            for item in new_products:
                product_diff.append(
                    dict_diff(dummy_product, item, fields=product_fields)
                )

            models.Log.objects.create(
                enquiry_id=instance.id,
                user=user,
                type='replace_enquiry_products_diff',
                data=json_dumps(product_diff),
            )

        return instance


class HeaderRetrieveUpdate(EnquiryObjectMixin, generics.RetrieveUpdateAPIView):
    serializer_class = serializers.Header
    permission_classes = (EntryPermission,)
    lookup_url_kwarg = 'enquiry_id'

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .select_related('manager', 'category')
            .prefetch_related('attachments')
            .annotate(
                enquiry_note=FilteredRelation(
                    'notes', condition=Q(notes__user=self.request.user)
                ),
                deadline_at=F('request__deadline_at'),
            )
            .annotate(note=F('enquiry_note__text'))
        )

    def perform_update(self, serializer):

        user = self.request.user
        old = serializers.Log(serializer.instance).data

        # Actual update
        instance = super().perform_update(serializer)

        new = serializers.Log(instance).data
        diff = dict_diff(old, new, show_unchanged=False)

        if diff:
            models.Log.objects.create(
                enquiry_id=instance.id,
                user=user,
                type='update_enquiry',
                data=json_dumps(diff),
            )

        # ------------------------------------------------------------------
        # Notification for a new assignee

        if (
            old['manager'] != new['manager']
            and new['manager'] is not None
            and new['manager']['id'] != self.request.user.id
        ):

            messages = notify_enquiry_assigned(instance.id, self.request.user)
            with mail.get_connection(fail_silently=True) as conn:
                conn.send_messages(messages)

        return instance


# ------------------------------------------------------------------------------


class EnquiryOptions(EnquiryObjectMixin, generics.GenericAPIView):
    permission_classes = (EntryPermission,)
    lookup_url_kwarg = 'enquiry_id'

    def get_queryset(self):
        return models.Enquiry.objects.permitted(self.request.user)

    def get(self, request, *args, **kwargs):

        user = request.user
        enquiry = self.object

        can_write = not is_readonly()
        can_make_progress = user.has_perm('api.progress_enquiry')

        has_rfx = models.Request.objects.filter(enquiry_id=enquiry.id).exists()

        can_publish = (
            enquiry.status == ES.DRAFT and can_make_progress and not has_rfx
        )

        can_make_link = 'api.update_enquiry' in enquiry.permissions(
            request.user
        )

        options = {
            'publish': can_publish,
            'cancel': can_make_progress and enquiry.status != ES.CLOSED,
            'deadline': (
                has_rfx
                and enquiry.status < ES.CLOSED
                and (enquiry.status == ES.DRAFT or can_make_progress)
            ),
            'clone': user.has_perm('api.create_enquiry'),
            'links': can_make_link,
            'access': can_make_progress,
        }

        for key in options:
            options[key] = options[key] and can_write

        return Response(options)


class EnquiryCancel(generics.GenericAPIView):
    permission_classes = (permissions.EnquiryClosePermission,)
    lookup_url_kwarg = 'enquiry_id'

    def get_queryset(self):
        return models.Enquiry.objects.permitted(self.request.user)

    def post(self, request, *args, **kwargs):

        user = request.user
        enquiry = self.object

        # ----------------------------------------------------------------------

        enqury_old = {
            'status': force_text(ES.i18n[enquiry.status]),
            'reason': force_text(ER.i18n[enquiry.reason]),
        }

        olds = list(
            models.Quote.objects.values_list('id', 'status', 'reason').filter(
                request__enquiry=enquiry
            )
        )

        # ----------------------------------------------------------------------

        models.Quote.objects.filter(request__enquiry=enquiry).update(
            status=QS.CLOSED, reason=QR.CANCELLED
        )

        enquiry.status = ES.CLOSED
        enquiry.reason = ER.CANCELLED
        enquiry.save(update_fields=('status', 'reason'))

        # ----------------------------------------------------------------------

        enqury_new = {
            'status': force_text(ES.i18n[ES.CLOSED]),
            'reason': force_text(ER.i18n[ER.CANCELLED]),
        }

        diff = dict_diff(enqury_old, enqury_new, show_unchanged=False)

        if diff:
            models.Log.objects.create(
                enquiry_id=enquiry.id,
                user=user,
                type='update_enquiry',
                data=json_dumps(diff),
            )

        # ----------------------------------------------------------------------

        new = {
            'status': force_text(QS.i18n[QS.CLOSED]),
            'reason': force_text(QR.i18n[QR.CANCELLED]),
        }

        messages = []

        for quote_id, status, reason in olds:
            old = {
                'status': force_text(ES.i18n[status]),
                'reason': force_text(ER.i18n[reason]),
            }

            messages.extend(
                quote_notify(user, quote_id, (status, QS.CLOSED), old, new)
            )

        with mail.get_connection(fail_silently=True) as conn:
            conn.send_messages(messages)

        # ----------------------------------------------------------------------

        return Response({'result': 'ok'})
