# coding: utf-8

import logging

from blackbox import Blackbox
from django.conf import settings
from django.contrib.auth.models import UserManager as BaseUserManager
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.db.models import Q
from django.apps import apps

from django.db.models.functions import Now
from django.db.models.query import ModelIterable
from rest_framework.exceptions import ValidationError

from procu.api.enums import QS
from procu.api.exceptions import FetchUserError, FetchUserNotFound
from procu.rest import serializers
from procu.soft_delete.models import ExistingOnlyManager

logger = logging.getLogger(__name__)


# ------------------------------------------------------------------------------


class UserinfoSerializer(serializers.Serializer):
    username = serializers.CharField()
    email = serializers.EmailField()
    first_name = serializers.CharField()
    last_name = serializers.CharField()
    language = serializers.CharField(max_length=2)
    sex = serializers.ChoiceField(choices=(0, 1, 2))


class UserManager(BaseUserManager):
    def get_or_create_by_login(self, username, update=False):

        if not update:
            try:
                return self.get(username=username, is_staff=True)
            except ObjectDoesNotExist:
                pass

        blackbox = getattr(settings, 'YAUTH_BLACKBOX_INSTANCE') or Blackbox()
        fields = dict(settings.YAUTH_PASSPORT_FIELDS)

        try:
            info = blackbox.userinfo(
                uid_or_login=username,
                userip='127.0.0.1',
                by_login=True,
                dbfields=fields,
                emails='getdefault',
            )

        except Exception:
            logger.exception('Error while fetching %s@', username)
            raise FetchUserError('Could not fetch %s@ from Blackbox' % username)

        if not info['uid']:
            raise FetchUserNotFound('Login %s@ not found' % username)

        user_fields = {
            model_field: info['fields'][model_field]
            for model_field in fields.values()
        }
        user_fields['email'] = info['default_email']

        serializer = UserinfoSerializer(data=user_fields)
        try:
            serializer.is_valid(raise_exception=True)

        except ValidationError as exc:
            logger.exception('Invalid response from Blackbox')
            raise FetchUserError(exc.detail)

        user, created = self.get_or_create(is_staff=True, username=username)

        if update or created:
            update_fields = list(serializer.validated_data.keys())
            update_fields.append('password')

            user.set_unusable_password()
            for field, value in serializer.validated_data.items():
                setattr(user, field, value)
            user.save(update_fields=update_fields)

        return user

    def get_by_natural_key(self, username):
        return self.get(username__iexact=username)


# ------------------------------------------------------------------------------


class QuoteManager(ExistingOnlyManager):
    def expired(self):
        qs = super().get_queryset()
        return qs.filter(
            models.Q(deadline_at__lt=Now(), status=QS.BIDDING)
            | models.Q(deadline_at__gt=Now(), status=QS.REVIEW)
        )

    def permitted(self, user):
        qs = super().get_queryset()

        EnquiryAccess = apps.get_model('api', 'EnquiryAccess')

        if user.is_staff:
            if not user.has_perm('api.all_enquiry'):
                es = EnquiryAccess.objects.filter(user=user)
                qs = qs.filter(request__enquiry__in=es.values('enquiry_id'))
            return qs

        return qs.filter(status__gt=QS.DRAFT, supplier_id=user.supplier_id)


# ------------------------------------------------------------------------------


class EnquiryManager(models.Manager):
    def permitted(self, user):
        qs = super().get_queryset()

        if user.is_staff:
            if not user.has_perm('api.all_enquiry'):
                qs = qs.filter(access__user=user)
            return qs

        return qs.none()


# ------------------------------------------------------------------------------


class RequestManager(models.Manager):
    def permitted(self, user):
        qs = super().get_queryset()

        Quote = apps.get_model('api', 'Quote')
        EnquiryAccess = apps.get_model('api', 'EnquiryAccess')

        if user.is_staff:
            if not user.has_perm('api.all_enquiry'):
                es = EnquiryAccess.objects.filter(user=user)
                qs = qs.filter(enquiry_id__in=es.values('enquiry_id'))
            return qs

        rfxs = Quote.objects.values('request_id').filter(
            supplier_id=user.supplier_id
        )
        return qs.filter(id__in=rfxs)


# ------------------------------------------------------------------------------


class EnquiryNoteManager(models.Manager):
    def permitted(self, user):
        qs = super().get_queryset()
        if user.is_staff:
            return qs.filter(user=user)
        return qs.none()


# ------------------------------------------------------------------------------


class InvoiceManager(ExistingOnlyManager):
    def permitted(self, user):
        qs = super().get_queryset()

        if user.is_staff:
            if not user.has_perm('api.all_enquiry'):
                qs = qs.filter(enquiry__access__user=user)
            return qs

        else:
            qs = qs.filter(supplier=user.supplier)

        return qs


# ------------------------------------------------------------------------------


class PreparedIterable(ModelIterable):
    def __iter__(self):
        base_iter = super().__iter__()
        config = self.queryset.config

        if config is None:
            for obj in base_iter:
                yield obj
        else:
            for obj in base_iter:
                obj.prepare(**config)
                yield obj


class PreparedQuerySet(models.query.QuerySet):
    config = None

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self._iterable_class = PreparedIterable

    def _clone(self, **kwargs):
        clone = super()._clone(**kwargs)
        clone.config = self.config
        return clone


class BaseQuoteProductManager(ExistingOnlyManager):
    _queryset_class = PreparedQuerySet

    def prepared(self, snapshot=None, **kwargs):

        snapshot_filter = Q(snapshot__isnull=True)

        if snapshot:
            snapshot_filter |= Q(snapshot__in=snapshot)

        qs = (
            self.get_queryset()
            .filter(snapshot_filter)
            .select_related('currency')
        )
        qs.config = kwargs
        return qs

    def permitted(self, user):
        qs = super().get_queryset()

        if user.is_staff:
            return qs

        return qs.filter(quote__supplier=user.supplier)


class QuoteProductManager(BaseQuoteProductManager):
    def get_queryset(self):
        return super().get_queryset().filter(snapshot__isnull=True)


class QuoteProductSummaryManager(BaseQuoteProductManager):
    _queryset_class = PreparedQuerySet


# ------------------------------------------------------------------------------


class EnquiryCommentManager(ExistingOnlyManager):
    def permitted(self, user):
        qs = super().get_queryset()

        if user.is_staff:
            if not user.has_perm('api.all_enquiry'):
                qs = qs.filter(enquiry__access__user=user)
            return qs

        return qs.none()


# ------------------------------------------------------------------------------


class QuoteCommentManager(ExistingOnlyManager):
    def permitted(self, user):
        qs = super().get_queryset()

        if user.is_staff:
            return qs

        return qs.filter(
            quote__status__gt=QS.DRAFT, quote__supplier_id=user.supplier_id
        )


# ------------------------------------------------------------------------------


class LogManager(models.Manager):
    def permitted(self, user):
        qs = super().get_queryset()

        if user.is_staff:
            if not user.has_perm('api.all_enquiry'):
                qs = qs.filter(enquiry__access__user=user)
            return qs

        return qs.none()
