# coding: utf-8
import os
from datetime import datetime, timedelta

import blackbox
from blackbox import Blackbox
from django.conf import settings
from django.db import connections
from django.db.migrations.loader import MigrationLoader
from django.utils.dateparse import parse_datetime
from redis import StrictRedis
from rest_framework import status, views
from rest_framework.response import Response

from procu.api import models, utils
from .permissions import MonitoringPermission


class MigrationStateView(views.APIView):
    authentication_classes = ()
    permission_classes = (MonitoringPermission,)

    @staticmethod
    def get(request):
        conn = connections['default']
        loader = MigrationLoader(conn, ignore_no_migrations=True)

        unapplied = set(loader.graph.nodes.keys()) - loader.applied_migrations
        unapplied = bool(unapplied)

        if unapplied:
            code = status.HTTP_503_SERVICE_UNAVAILABLE
        else:
            code = status.HTTP_200_OK

        return Response({'all_applied': not unapplied}, status=code)


class StatsView(views.APIView):
    authentication_classes = ()
    permission_classes = (MonitoringPermission,)

    @staticmethod
    def get(request):

        r = StrictRedis(**settings.CELERY_REDIS)
        backlog_size = r.llen('celery')

        result = {
            "sensors": [
                {
                    "labels": {"sensor": "celery_backlog_size"},
                    "value": backlog_size,
                },
                {
                    "labels": {"sensor": "suppliers_total"},
                    "value": models.Supplier.objects.filter(
                        is_deleted=False
                    ).count(),
                },
                {
                    "labels": {"sensor": "users_total"},
                    "value": models.User.objects.filter(
                        is_deleted=False
                    ).count(),
                },
                {
                    "labels": {"sensor": "users_internal"},
                    "value": models.User.objects.filter(
                        is_deleted=False, is_staff=True
                    ).count(),
                },
                {
                    "labels": {"sensor": "users_external"},
                    "value": models.User.objects.filter(
                        is_deleted=False, is_staff=False
                    ).count(),
                },
            ]
        }

        return Response(result)


class StateView(views.APIView):
    authentication_classes = ()

    @staticmethod
    def test_blackbox():
        bb = getattr(settings, 'YAUTH_BLACKBOX_INSTANCE') or blackbox.Blackbox()
        try:
            bb.userinfo(
                uid_or_login=settings.ROBOT_LOGIN,
                userip='127.0.0.1',
                by_login=True,
            )
        except Exception:
            return False

        return True

    def get(self, request):

        state = 'OK'
        status_code = status.HTTP_200_OK

        if not self.test_blackbox():
            state = 'FAILED_BLACKBOX'
            status_code = 599

        payload = {'state': state}

        if utils.is_internal(request):
            image = os.getenv('DEPLOY_DOCKER_IMAGE', ':')
            version = image.split(':')[1]

            payload['version'] = version

        return Response(payload, status=status_code)


class TokenExpireView(views.APIView):
    authentication_classes = ()
    permission_classes = (MonitoringPermission,)

    @staticmethod
    def get_token_state():

        bb = Blackbox()

        resp = bb.oauth(
            settings.INTRANET_OAUTH_TOKEN,
            '1.1.1.1',
            by_token=True,
            emails='getdefault',
        )

        oauth = resp.get('oauth')

        if oauth is None:
            return 'expired'

        expire_time = parse_datetime(str(oauth.get('expire_time')))

        if expire_time is None:
            return 'error'

        delta = expire_time - datetime.now()

        if delta < timedelta(days=7):
            return 'expire_soon'

        return 'ok'

    def get(self, request):

        state = self.get_token_state()

        if state == 'ok':
            code = status.HTTP_200_OK
        else:
            code = status.HTTP_503_SERVICE_UNAVAILABLE

        return Response({'token': state}, status=code)
