# coding: utf-8

from django.utils.translation import gettext_lazy as _

from procu.api import models
from procu.api.common.fields import EmployeeField, TicketsField
from procu.api.enquiry.invoice.serializers import EnumPK
from procu.rest import serializers
from procu.rest.exceptions import ValidationError
from ..serializers import (
    CFOBrief,
    MVPBrief,
    ProjectBrief,
    PurchaseGroupBrief,
    ServiceBrief,
    TaskBrief,
)


class Currency(serializers.ModelSerializer):
    class Meta:
        model = models.Currency
        fields = ('id', 'prefix', 'suffix', 'char_code')


class LegalEntity(serializers.ModelSerializer):
    orgid = serializers.CharField(source='oracle_legal_entity.orgid')
    shortname = serializers.CharField(source='oracle_legal_entity.shortname')
    name = serializers.CharField(source='oracle_legal_entity.name')

    class Meta:
        model = models.LegalEntity
        fields = ('id', 'orgid', 'shortname', 'name')


class Product(serializers.Serializer):
    name = serializers.CharField(
        max_length=255, label=_('PURCHASE_FORM::PRODUCT_NAME')
    )
    price = serializers.MonetaryField(
        label=_('PURCHASE_FORM::PRODUCT_PRICE'), allow_null=False
    )
    currency = EnumPK(
        queryset=models.Currency.objects.filter(is_deleted=False),
        display_field='char_code',
        serializer=Currency,
        label=_('PURCHASE_FORM::PRODUCT_CURRENCY'),
    )
    qty = serializers.IntegerField(label=_('PURCHASE_FORM::PRODUCT_QTY'))
    url = serializers.URLField(
        max_length=511, label=_('PURCHASE_FORM::PRODUCT_URL')
    )


class Purchase(serializers.Serializer):

    subject = serializers.CharField(
        label=_('PURCHASE_FORM::SUBJECT'),
        max_length=255,
        style={'x-hint': _('PURCHASE_FORM::SUBJECT_HINT')},
    )
    is_urgent = serializers.BooleanField(label=_('PURCHASE_FORM::IS_URGENT'))
    reason = serializers.CharField(
        label=_('PURCHASE_FORM::REASON'),
        required=False,
        allow_blank=True,
        style={'x-required': True},
    )
    description = serializers.CharField(label=_('PURCHASE_FORM::DESCRIPTION'))

    products = serializers.ListSerializer(
        child=Product(), max_length=1, label=_('PURCHASE_FORM::PRODUCTS')
    )

    expected_on = serializers.DateField(label=_('PURCHASE_FORM::EXPECTED_ON'))
    term = serializers.CharField(
        label=_('PURCHASE_FORM::TERM'), max_length=255, allow_blank=True
    )

    legal_entity = serializers.PKPrettyField(
        queryset=models.LegalEntity.objects.filter(
            oracle_legal_entity__isnull=False, is_deleted=False
        ),
        serializer=LegalEntity,
        label=_('PURCHASE_FORM::LEGAL_ENTITY'),
    )

    mvp = serializers.PKPrettyField(
        queryset=models.OracleMVP.objects.filter(is_deleted=False),
        serializer=MVPBrief,
        label=_('PURCHASE_FORM::MVP'),
    )

    initiator = EmployeeField(label=_('PURCHASE_FORM::INITIATOR'))
    cfo = serializers.PKPrettyField(
        queryset=models.OracleCFO.objects.filter(is_deleted=False),
        serializer=CFOBrief,
        allow_null=False,
        label=_('PURCHASE_FORM::CFO'),
    )

    project = serializers.PKPrettyField(
        queryset=models.OracleProject.objects.filter(is_deleted=False),
        serializer=ProjectBrief,
        allow_null=True,
        label=_('PURCHASE_FORM::PROJECT'),
    )

    task = serializers.PKPrettyField(
        queryset=models.OracleTask.objects.filter(is_deleted=False),
        serializer=TaskBrief,
        allow_null=True,
        label=_('PURCHASE_FORM::TASK'),
    )

    service = serializers.PKPrettyField(
        queryset=models.OracleService.objects.filter(is_deleted=False),
        serializer=ServiceBrief,
        allow_null=True,
        label=_('PURCHASE_FORM::SERVICE'),
    )

    purchase_group = serializers.PKPrettyField(
        queryset=models.OraclePurchaseGroup.objects.filter(is_deleted=False),
        serializer=PurchaseGroupBrief,
        allow_null=True,
        label=_('PURCHASE_FORM::PURCHASE_GROUP'),
    )

    tickets = TicketsField(allow_empty=False, label=_('PURCHASE_FORM::TICKETS'))

    def validate(self, data):

        if not data['products']:
            raise ValidationError(
                {'reason': _('PURCHASE_FORM::ERROR_NO_PRODUCTS')}
            )

        if data['is_urgent'] and not data.get('reason'):
            raise ValidationError({'reason': _('PURCHASE_FORM::ERROR_REASON')})

        return data
