# coding: utf-8

import logging

from django.conf import settings
from django.template import loader
from rest_framework.response import Response

from procu.api.enums import LINK
from procu.api.utils import get_tracker_client
from procu.rest import exceptions, generics
from procu.rest.metadata.mixins import CreateMetaMixin
from . import serializers
from .utils import OEBSError, get_approvers

logger = logging.getLogger(__name__)


PROBLEMS = {
    'le_project': 'Проект не соответствует юрлицу',
    'project_task': 'Задача не соответствует проекту',
    'le_task': 'Задача не соответствует юрлицу',
}


class Purchase(CreateMetaMixin, generics.GenericAPIView):
    serializer_class = serializers.Purchase
    permission_classes = ()

    def post(self, request, *args, **kwargs):

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        data = serializer.data

        data['description'] = '\n'.join(
            f'  ##{line}##'
            for line in filter(None, data['description'].split('\n'))
        )

        if data['is_urgent']:
            data['reason'] = '\n'.join(
                f'  ##{line}##'
                for line in filter(None, data['reason'].split('\n'))
            )

        subject = f'Оплата корп. картой: {data["subject"]}'

        body = loader.render_to_string('oebs/scs.txt', context={'data': data})
        body = '\n'.join(filter(None, body.split('\n')))

        issue_kwargs = dict(
            queue=settings.SCS_QUEUE,
            type={'name': 'Task'},
            summary=subject,
            description=body,
            tags=['procu'],
            createdBy=self.request.user.username,
        )

        if data['is_urgent']:
            issue_kwargs['priority'] = 'critical'

        for field, getter in settings.SCS_FIELDS.items():
            try:
                issue_kwargs[field] = getter(data)
            except (KeyError, IndexError, TypeError, AttributeError):
                pass

        client = get_tracker_client()

        try:
            scs_issue = client.issues.create(**issue_kwargs)
            url = settings.LINK_URLS[LINK.TRACKER].format(key=scs_issue.key)

        except Exception:
            logger.exception('Could not create ticket')
            raise exceptions.ValidationError(
                {'detail': 'Could not create SCS ticket'}
            )

        # ------------------------------------------
        # Validate data

        problems = []

        objects = serializer.validated_data

        legal_entity = objects['legal_entity'].oracle_legal_entity
        project = objects['project']
        task = objects['task']

        if project:
            if project.company_id != legal_entity.id:
                problems.append('le_project')

            if task and task.project_id != project.id:
                problems.append('project_task')

                if (
                    not {'le_project', 'project_task'}.issubset(problems)
                    and task.project.company_id != project.company_id
                ):
                    problems.append('le_task')

        if problems:
            problem_list = '\n'.join(f'* {PROBLEMS[p]}' for p in problems)

            scs_issue.comments.create(
                text=(
                    f'Во введённых данных обнаружены следующие ошибки:\n'
                    f'{problem_list}'
                )
            )

        # ------------------------------------------
        # Fetch approvers

        appr_kwargs = {
            'org_id': data['legal_entity']['orgid'],
            'cfo': data['cfo']['key'],
        }

        if data['project']:
            appr_kwargs['project'] = data['project']['key']

        if data['task']:
            appr_kwargs['task'] = data['task']['key']

        if data['mvp']:
            appr_kwargs['mvp'] = data['mvp']['key']

        if data['service']:
            appr_kwargs['service'] = data['service']['key']

        try:
            approvers = get_approvers(**appr_kwargs)

            lines = [
                'Требуется подтверждение от одного из ответственных менеджеров:'
            ]
            lines.extend((f'* кто:{username}' for username in approvers))

            text = '\n'.join(lines)

            if approvers:
                scs_issue.comments.create(text=text)

        except OEBSError as exc:
            logger.exception('Could not fetch approvers: %s', exc.args[0])

        # ------------------------------------------

        return Response(
            {'tickets': [{'key': scs_issue.key, 'url': url, 'title': subject}]}
        )
