# coding: utf-8

from procu.api import models
from procu.api.common.comment import create_quote_comment
from procu.api.utils import dict_diff, is_readonly, json_dumps
from procu.rest import generics
from procu.rest.permissions import ExternalOnly
from . import serializers
from .permissions import CreatePermission, UpdateDestroyPermission


class CommentListCreate(generics.ListCreateAPIView):
    permission_classes = (ExternalOnly, CreatePermission)
    output_serializer_class = serializers.RetrieveUpdate

    def get_queryset(self):
        queryset = (
            models.QuoteComment.objects.permitted(self.request.user)
            .filter(quote=self.kwargs['quote_id'])
            .select_related('author')
            .prefetch_related('attachments')
            .order_by('-id')
        )

        if is_readonly():
            queryset = queryset.extra(select={'can_edit': 'false'})
        else:
            queryset = queryset.extra(
                select={'can_edit': 'author_id = %s'},
                select_params=(self.request.user.id,),
            )

        return queryset

    def get_serializer_class(self):
        if self.request.method == 'POST':
            return serializers.Create
        else:
            return serializers.List

    def perform_create(self, serializer):
        data = serializer.data

        return create_quote_comment(
            quote_id=int(self.kwargs['quote_id']),
            user=self.request.user,
            data={'message': data['message']},
            attachment_ids=data['attachments'],
        )


class CommentRetrieveUpdateDestroy(generics.RetrieveUpdateDestroyAPIView):
    serializer_class = serializers.RetrieveUpdate
    permission_classes = (ExternalOnly, UpdateDestroyPermission)
    lookup_url_kwarg = 'comment_id'

    def get_queryset(self):
        return models.QuoteComment.objects.permitted(self.request.user).filter(
            quote=self.kwargs['quote_id']
        )

    def perform_update(self, serializer):

        old = serializers.Log(serializer.instance).data
        instance = serializer.save()

        new = serializers.Log(instance).data

        diff = dict_diff(old, new)

        if diff:
            quote_id = instance.quote_id
            enquiry_id = models.Request.objects.values_list(
                'enquiry_id', flat=True
            ).get(quotes=quote_id)

            models.Log.objects.create(
                quote_id=quote_id,
                enquiry_id=enquiry_id,
                user=self.request.user,
                type='update_quote_comment',
                data=json_dumps(diff),
            )

    def perform_destroy(self, instance):
        old = serializers.Log(instance).data
        super().perform_destroy(instance)

        quote_id = instance.quote_id
        enquiry_id = models.Request.objects.values_list(
            'enquiry_id', flat=True
        ).get(quotes=quote_id)

        models.Log.objects.create(
            quote_id=quote_id,
            enquiry_id=enquiry_id,
            user=self.request.user,
            type='remove_quote_comment',
            old=json_dumps(old),
        )
