# coding: utf-8
from functools import partial

from django.utils.translation import gettext_lazy as _
from yarl import URL

from procu.api import models
from procu.api.address.serializers import FullName
from procu.api.common.fields import AttachmentField
from procu.rest import serializers


# ------------------------------------------------------------------------------


def lazy_field(field, *args, **kwargs):
    return field(*args, **kwargs)


class InfoFields(object):
    address = partial(lazy_field, label=_('ENQUIRY_EXT_INFO::DELIVER_TO'))
    legal_entity = partial(
        lazy_field, label=_('ENQUIRY_EXT_INFO::LEGAL_ENTITY')
    )
    comment = partial(
        serializers.CharField,
        source='description',
        label=_('ENQUIRY_EXT_INFO::COMMENT'),
    )
    attachments = partial(
        AttachmentField, label=_('ENQUIRY_EXT_INFO::ATTACHMENTS'), many=True
    )


# ------------------------------------------------------------------------------


class LegalEntityBrief(serializers.ModelSerializer):
    class Meta:
        model = models.LegalEntity
        fields = ('title',)


class LegalEntityFull(serializers.ModelSerializer):
    class Meta:
        model = models.LegalEntity
        fields = ('title', 'details')


# ------------------------------------------------------------------------------


def maybe_get_url_by_address(obj):
    if obj.geo:
        try:
            url = URL('https://yandex.ru/maps/')
            url = url.with_query(
                **{
                    'l': 'map',
                    'mode': 'whatshere',
                    'whatshere[point]': f'{obj.geo["lon"]},{obj.geo["lat"]}',
                    'whatshere[zoom]': 17,
                }
            )
            return str(url)

        except KeyError:
            pass


class AddressBrief(serializers.ModelSerializer):

    data = serializers.SerializerMethodField(
        label=_('ENQUIRY_EXT_INFO::ADDRESS')
    )

    class Meta:
        model = models.Address
        fields = ('data',)

    @staticmethod
    def get_data(obj):
        return {'text': obj.text, 'url': maybe_get_url_by_address(obj)}


class ContactSerializer(serializers.Serializer):

    name = FullName(label=_('ADDRESS_CONTACT::NAME'))
    is_staff_phone = serializers.BooleanField(
        required=False, label=_('ADDRESS_CONTACT::IS_STAFF_PHONE')
    )
    phone = serializers.CharField(label=_('ADDRESS_CONTACT::PHONE'))

    class Meta:
        fields = ('name', 'is_staff_phone', 'phone')


class AddressFull(serializers.ModelSerializer):
    contacts = ContactSerializer(many=True, required=False)
    data = serializers.SerializerMethodField(
        label=_('ENQUIRY_EXT_INFO::ADDRESS')
    )

    class Meta:
        model = models.Address
        fields = ('contacts', 'comment', 'data')

    @staticmethod
    def get_data(obj):
        return {'text': obj.text, 'url': maybe_get_url_by_address(obj)}
