# coding: utf-8

from rest_framework.response import Response

from procu.api import models
from procu.api.enums import QS
from procu.rest import generics, permissions
from . import serializers

INFO = {
    QS.SHIPPED: {
        'text': 'Заказ согласован и размещён. '
        'Пожалуйста, сообщите дату поставки, если она изменилась. '
        'Также просим прислать сканы закрывающих документов не позже '
        'чем за день до поставки.'
    }
}


class Statuses(generics.GenericAPIView):
    lookup_url_kwarg = 'quote_id'
    permission_classes = (permissions.ExternalOnly,)

    def get_queryset(self):
        return models.Quote.objects.permitted(self.request.user)

    def get(self, request, quote_id):
        quote = self.object

        statuses = []

        for (status, key), name in zip(QS.keys.items(), QS.i18n.values()):
            if status == QS.DRAFT:
                continue

            statuses.append(
                {'key': key, 'name': name, 'is_current': status == quote.status}
            )

        return Response(statuses)


class Info(generics.RetrieveAPIView):
    lookup_url_kwarg = 'quote_id'
    permission_classes = (permissions.ExternalOnly,)

    def get_queryset(self):
        return models.Quote.objects.permitted(self.request.user)

    def get_serializer_class(self):
        quote = self.object
        return {
            QS.BIDDING: serializers.NoInfo,
            QS.REVIEW: serializers.NoInfo,
            QS.CHECKOUT: serializers.NoInfo,
            QS.SHIPPED: serializers.TextInfo,
            QS.CLOSED: serializers.NoInfo,
        }[quote.status]

    def retrieve(self, request, quote_id):
        quote = self.object

        data = INFO.get(quote.status, {})

        serializer = self.get_serializer(data=data)
        serializer.is_valid(raise_exception=True)

        return Response(serializer.validated_data)


class Feedback(generics.RetrieveUpdateAPIView):
    lookup_url_kwarg = 'quote_id'
    permission_classes = (permissions.ExternalOnly,)

    def get_queryset(self):
        return models.Quote.objects.permitted(self.request.user)

    def get_serializer_class(self):
        quote = self.object
        return {
            QS.BIDDING: serializers.NoFeedback,
            QS.REVIEW: serializers.NoFeedback,
            QS.CHECKOUT: serializers.NoFeedback,
            QS.SHIPPED: serializers.ShippedFeedback,
            QS.CLOSED: serializers.NoFeedback,
        }[quote.status]
