# coding: utf-8

from collections import defaultdict
from datetime import datetime
from decimal import Decimal
from math import sqrt
from statistics import mean, variance

from django.db.models import Q, Sum
from django.db import transaction

from procu.api import models, utils
from procu.api.enums import ER, ES, SavingNote
from procu.api.utils import format_price


def get_enquiry_saving(rfx):
    # See https://st.yandex-team.ru/PROCU-350 for details

    if rfx.status not in (ES.CLOSED, ES.SHIPPED):
        return {}

    currency = models.Currency.objects.get(char_code='RUB')

    obj = {}

    n_products = len(rfx.products.values_list('id', flat=True))

    qs = models.Quote.objects.filter(request=rfx, is_deleted=False).values(
        'id', 'has_won', 'reason'
    )

    if rfx.status == ES.CLOSED and not any(
        q['reason'] == ER.DELIVERED for q in qs
    ):
        return {}

    n_winners = sum((q['has_won'] for q in qs), 0)

    if n_winners < 1:
        obj['note'] = SavingNote.NO_WINNERS
        return obj

    if n_winners > 1:
        obj['note'] = SavingNote.MULTIPLE_WINNERS
        return obj

    winner = next(q['id'] for q in qs if q['has_won'])

    qps = models.QuoteProduct.objects.prepared(
        totals=True, currency=currency
    ).filter(
        is_deleted=False,
        is_replacement=False,
        price__isnull=False,
        price__gt=0,
        quote__request=rfx,
    )

    offers = defaultdict(list)

    for qp in qps:
        offers[qp.quote_id].append(qp)

    offers = {
        quote: offer
        for quote, offer in offers.items()
        if len(offer) == n_products
    }

    if len(offers) < 2:
        obj['note'] = SavingNote.LESS_THAN_TWO_OFFERS
        return obj

    if winner not in offers:
        obj['note'] = SavingNote.NON_TRIVIAL_WINNER
        return obj

    totals = {}

    for quote_id, offer in offers.items():
        total = sum((qp.total for qp in offer), Decimal('0.00'))
        totals[quote_id] = total

    total_mean = mean(totals.values())
    total_var = sqrt(variance(totals.values(), total_mean))

    if total_var > total_mean:
        obj['note'] = SavingNote.INVALID_PRICES
        return obj

    value = total_mean - totals[winner]

    if value < 0.:
        obj['note'] = SavingNote.NEGATIVE_SAVING
        return obj

    obj.update(
        {
            'saving': utils.money(value),
            'currency': currency,
            'note': SavingNote.NONE,
            'best_price': utils.money(totals[winner]),
            'expected_price': utils.money(total_mean),
        }
    )

    return obj


# Date of the very first release of Yandex.Procurements.
# Before that it is unclear whether a closed enquiry
# is a completed or a cancelled one.
THE_VERY_BEGINNING = datetime(2017, 4, 24)


def update_savings(*, force=False):

    if force:
        # Process all closed enquiries
        condition = Q(status=ES.CLOSED)
    else:
        # Skip closed enquiries processed earlied
        condition = Q(status=ES.CLOSED, enquiry__saving__isnull=True)

    condition |= Q(status=ES.SHIPPED)

    with transaction.atomic():
        rfxs = models.Request.objects.filter(
            condition, enquiry__created_at__gt=THE_VERY_BEGINNING
        ).order_by('-id')

        for rfx in rfxs:
            obj = get_enquiry_saving(rfx)
            if obj:
                models.EnquirySaving.objects.update_or_create(
                    enquiry=rfx.enquiry, defaults=obj
                )

        models.EnquirySaving.objects.exclude(
            enquiry__status__in=(ES.CLOSED, ES.SHIPPED)
        ).delete()


def get_total_saving():

    saving = models.EnquirySaving.objects.filter(
        saving__isnull=False
    ).aggregate(best=Sum('best_price'), expected=Sum('expected_price'))

    absolute = saving['expected'] - saving['best']

    try:
        relative = Decimal('1.00') - saving['best'] / saving['expected']
    except ZeroDivisionError:
        relative = 0.

    absolute = format_price(absolute)

    prefix, suffix = models.Currency.objects.values_list(
        'prefix', 'suffix'
    ).get(char_code='RUB')

    absolute = f'{prefix}\u00A0{absolute}\u00A0{suffix}'.strip()
    relative = f'{round(relative * 100, 2)}%'

    return {'absolute': absolute, 'relative': relative}
