# coding: utf-8

from django.db.models import F
from django.db.models.expressions import RawSQL
from django_filters.rest_framework import DjangoFilterBackend

from procu.api import models
from procu.api.enquiry.filters import YPSearchFilter
from procu.rest import filters, generics, pagination, permissions
from .filters import EnquiryFilterSet, SavingOrderingFilter
from .permissions import SavingsPermissions
from .serializers import EnquirySerializer


class EnquirySavingList(generics.ListAPIView):
    pagination_class = pagination.PageNumberPagination
    permission_classes = (SavingsPermissions, permissions.ListPermission)
    serializer_class = EnquirySerializer
    filter_class = EnquiryFilterSet
    filter_backends = (
        DjangoFilterBackend,
        YPSearchFilter,
        filters.SearchFilter,
        SavingOrderingFilter,
    )

    ordering_fields = (
        'id',
        'created_at,',
        'updated_at',
        'subject',
        'absolute',
        'relative',
    )
    ordering = ('-updated_at',)

    search_fields = ('id', 'subject', 'links__key', 'comment')

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .filter(saving__isnull=False)
            .annotate(
                absolute=F('saving__saving'),
                note=F('saving__note'),
                currency=F('saving__currency_id'),
                currency_prefix=F('saving__currency__prefix'),
                currency_suffix=F('saving__currency__suffix'),
                relative=RawSQL(
                    'round((1 - api_enquirysaving.best_price '
                    '/ api_enquirysaving.expected_price) * 100, 2)',
                    (),
                ),
            )
            .select_related('saving', 'saving__currency')
        )
