# coding: utf-8

from django.http import HttpResponse
from django.utils.translation import gettext as _
from rest_framework.exceptions import NotFound
from rest_framework.response import Response

from procu.api.spark.utils import NoData, get_risks_report, get_risks_summary
from procu.rest import generics
from procu.rest.permissions import StaffOnly
from .serializers import QueryParamsSerializer, Report


class RiskSummary(generics.GenericAPIView):
    serializer_class = Report
    permission_classes = (StaffOnly,)
    query_params_serializer_class = QueryParamsSerializer

    def get(self, request):
        query_params = self.query_params
        vat_id = query_params['vat_id']

        try:
            summary = get_risks_summary(vat_id)
        except NoData:
            raise NotFound(_('SPARK_RISKS::NOT_FOUND'))

        serializer = self.get_serializer(data=summary)
        serializer.is_valid(raise_exception=True)
        return Response(serializer.validated_data)


class RiskReport(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    query_params_serializer_class = QueryParamsSerializer

    def get(self, request):
        query_params = self.query_params
        vat_id = query_params['vat_id']

        try:
            data = get_risks_report(vat_id)
        except NoData:
            raise NotFound(_('SPARK_RISKS::NOT_FOUND'))

        response = HttpResponse(data, content_type='application/pdf')

        response['Content-Disposition'] = (
            "inline; filename*=UTF-8''risks_%s.pdf" % vat_id
        )

        return response
