# coding: utf-8

from django.db.models import BooleanField, CharField, Count, F, Q, Value
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework import permissions
from rest_framework.filters import OrderingFilter, SearchFilter
from rest_framework.permissions import IsAuthenticated

from procu.api import enums, models
from procu.rest import generics, pagination
from procu.rest.filters import ExcludeFilter, IncludeFilter, ShowDeletedFilter
from procu.rest.permissions import StaffOnly
from .filters import HideFilter, OEBSLegalEntitiesOnly, SuppliersByTagsFilter
from ..mixins import SuggestMixin


class SupplierTagView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (IsAuthenticated,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (SearchFilter, IncludeFilter, ExcludeFilter)

    search_fields = ('label',)

    def get_queryset(self):
        user = self.request.user

        queryset = models.SupplierTag.objects.values(
            'id',
            subtitle=F('comment'),
            title=F('label'),
            type=Value('default', output_field=CharField()),
            directory=Value('supplier-tag', output_field=CharField()),
        ).order_by('label', 'id')

        if not user.is_staff:
            queryset = queryset.filter(is_hidden=False)

        return queryset


class AddressView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('text', 'label')

    queryset = models.Address.objects.values(
        'id',
        'is_deleted',
        title=F('label'),
        subtitle=F('text'),
        type=Value('default', output_field=CharField()),
        directory=Value('address', output_field=CharField()),
    ).order_by('is_deleted', 'id')


class LegalEntityView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
        OEBSLegalEntitiesOnly,
        DjangoFilterBackend,
    )

    filter_fields = ('can_pay_by_card',)

    search_fields = ('title', 'details')

    queryset = models.LegalEntity.objects.values(
        'id',
        'title',
        'is_deleted',
        type=Value('default', output_field=CharField()),
        directory=Value('legal-entity', output_field=CharField()),
    ).order_by('is_deleted', 'title', 'id')


class CurrencyView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (permissions.IsAuthenticated,)

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('char_code', 'prefix', 'suffix', 'name')

    queryset = models.Currency.objects.values(
        'id',
        'name',
        'char_code',
        'is_deleted',
        type=Value('currency', output_field=CharField()),
        directory=Value('currency', output_field=CharField()),
    ).order_by('is_deleted', 'id')


class SupplierView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
        SuppliersByTagsFilter,
        DjangoFilterBackend,
    )
    search_fields = (
        'title',
        'legal_name',
        'comment',
        'vat_id',
        'info',
        'agents__email',
        'tags__label',
    )

    filter_fields = ('vat_id',)

    queryset = models.Supplier.objects.values(
        'id',
        'title',
        'is_deleted',
        # Flags
        'risk',
        'is_cold',
        'can_pay_by_card',
        has_warnings=Count(F('warnings'), filter=Q(warnings__is_deleted=False)),
        # ---------------
        type=Value('default', output_field=CharField()),
        directory=Value('supplier', output_field=CharField()),
    ).order_by('is_deleted', 'title', 'id')

    @staticmethod
    def finalize_queryset(queryset):
        return queryset.distinct()

    def make_response_data(self, objects):

        for supplier in objects:
            flags = supplier['flags'] = []

            if supplier['risk'] == enums.RISK.HIGH:
                flags.append('high_risk')

            if supplier['has_warnings']:
                flags.append('warnings')

            if supplier['is_cold']:
                flags.append('cold')

            if supplier['can_pay_by_card']:
                flags.append('card')

            for field in ('risk', 'has_warnings', 'is_cold', 'can_pay_by_card'):
                supplier.pop(field, None)

        return super().make_response_data(objects)


class EnquiryCategoryView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name',)

    queryset = models.EnquiryCategory.objects.values(
        'id',
        'is_deleted',
        title=F('name'),
        type=Value('default', output_field=CharField()),
        directory=Value('category', output_field=CharField()),
    ).order_by('is_deleted', 'id')


# ------------------------------------------------------------------------------


class SupplierMixed(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        OrderingFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = (
        'title',
        'legal_name',
        'comment',
        'vat_id',
        'info',
        'agents__email',
    )

    queryset = models.Supplier.objects.values(
        'id',
        'title',
        'is_deleted',
        # Flags
        'risk',
        'is_cold',
        'can_pay_by_card',
        has_warnings=Count(F('warnings'), filter=Q(warnings__is_deleted=False)),
        # ---------------
        tag=Value(False, output_field=BooleanField()),
        type=Value('default', output_field=CharField()),
        directory=Value('supplier', output_field=CharField()),
    ).order_by('is_deleted', 'title', 'id')

    qs_tags = models.SupplierTag.objects.values(
        'id',
        title=F('label'),
        subtitle=F('comment'),
        tag=Value(True, output_field=BooleanField()),
        type=Value('default', output_field=CharField()),
        icon=Value('tag', output_field=CharField()),
        directory=Value('supplier-tag', output_field=CharField()),
    ).order_by('label', 'id')

    class TagViewMocked(object):
        search_fields = ('label',)

    def get(self, request, *args, **kwargs):

        qs_tags = self.qs_tags

        for backend in (SearchFilter, HideFilter):
            qs_tags = backend().filter_queryset(
                self.request, qs_tags, self.TagViewMocked()
            )

        qs_suppliers = list(self.filter_queryset(self.get_queryset()))

        for supplier in qs_suppliers:
            flags = supplier['flags'] = []

            if supplier['risk'] == enums.RISK.HIGH:
                flags.append('high_risk')

            if supplier['has_warnings']:
                flags.append('warnings')

            if supplier['is_cold']:
                flags.append('cold')

            if supplier['can_pay_by_card']:
                flags.append('card')

            for field in ('risk', 'has_warnings', 'is_cold', 'can_pay_by_card'):
                supplier.pop(field, None)

        objs = list(qs_tags)
        objs.extend(qs_suppliers)

        page = self.paginate_queryset(objs)
        return self.get_paginated_response(page)
