# coding: utf-8

import logging
from collections import OrderedDict

from django.db.models import F

from procu.api import models
from procu.api.enquiry.filters import YPSearchFilter
from procu.api.enums import QS
from procu.api.log.constants import LOG_GROUPS
from procu.rest import filters, generics, pagination
from procu.rest.permissions import StaffOnly
from . import serializers
from ..mixins import SuggestMixin

logger = logging.getLogger(__name__)


class StatusView(generics.GenericAPIView):
    pagination_class = pagination.NoPagination
    query_params_serializer_class = serializers.StatusConfigSerializer
    authentication_classes = ()

    def get(self, request):
        config = self.query_params

        data = []

        for code, name in QS.keys.items():

            if config['scope'] == 'external' and code == QS.DRAFT:
                continue

            item = OrderedDict(
                [
                    ('id', name),
                    ('name', QS.i18n[code]),
                    (
                        'is_active',
                        code
                        in (QS.BIDDING, QS.REVIEW, QS.CHECKOUT, QS.SHIPPED),
                    ),
                ]
            )

            data.append(item)

        return self.get_paginated_response(data)


class LogGroupView(generics.GenericAPIView):
    pagination_class = pagination.NoPagination

    def get(self, request):

        groups = zip(LOG_GROUPS.keys.values(), LOG_GROUPS.i18n.values())

        ids = request.GET.getlist('id')
        if ids:
            groups = filter(lambda x: x[0] in ids, groups)

        data = [
            {'id': key, 'title': name, 'type': 'default'}
            for key, name in groups
        ]

        return self.get_paginated_response(data)


class EnquiryView(SuggestMixin, generics.ListAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        YPSearchFilter,
        filters.IncludeFilter,
        filters.ExcludeFilter,
    )
    search_fields = ('id', 'subject')

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .values('id', 'key', 'subject')
            .order_by('-updated_at', 'id')
        )

    @staticmethod
    def finalize_queryset(qs):
        return qs.distinct('updated_at', 'id')

    def make_response_data(self, objects):
        output = []

        for obj in objects:
            output.append(
                {
                    'id': obj['id'],
                    'title': '{}: {}'.format(obj['key'], obj['subject']),
                    'type': 'default',
                }
            )

        return super().make_response_data(output)


class QuoteView(SuggestMixin, generics.ListAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (filters.IncludeFilter, filters.ExcludeFilter)
    search_fields = ('id', 'title')

    def get_queryset(self):

        rfx = generics.get_object_or_404(
            models.Request.objects.permitted(self.request.user),
            enquiry_id=self.kwargs['enquiry_id'],
        )

        return (
            rfx.quotes.values(title=F('supplier__title'))
            .annotate(id=F('supplier_id'))
            .order_by('status', 'id')
        )
