# coding: utf-8

from functools import partial

from django.core.exceptions import ObjectDoesNotExist
from django.db.models import (
    CharField,
    ExpressionWrapper,
    F,
    IntegerField,
    Value,
)
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework.filters import SearchFilter

from procu.api import models
from procu.api.utils import annotate, colour_gradient
from procu.rest import generics, pagination
from procu.rest.filters import ExcludeFilter, IncludeFilter, ShowDeletedFilter
from procu.rest.permissions import StaffOnly
from .filters import ProgramFilter, ProjectFilter, TaskFilter
from ..mixins import SuggestMixin

gradient = partial(colour_gradient, start_hex='#FFFFFF', finish_hex='#4dbd33')


class CFOView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('key', 'name')

    queryset = models.OracleCFO.objects.values(
        'id',
        title=F('key'),
        subtitle=F('name'),
        type=Value('default', output_field=CharField()),
    ).order_by('key', 'id')


class ProgramView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
        DjangoFilterBackend,
    )
    filter_class = ProgramFilter

    search_fields = ('key', 'name')

    queryset = models.OracleProgram.objects.values(
        'id',
        title=F('key'),
        subtitle=F('name'),
        type=Value('default', output_field=CharField()),
    ).order_by('key', 'id')


class ProjectView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
        DjangoFilterBackend,
    )
    filter_class = ProjectFilter

    search_fields = ('key', 'name')

    queryset = models.OracleProject.objects.order_by('key', 'id')

    def finalize_queryset(self, queryset):

        request = self.request

        expr = []

        try:
            legal_entity = int(request.GET.get('by_legal_entity', None))

            company_id, = models.LegalEntity.objects.values_list(
                'oracle_legal_entity_id'
            ).get(id=legal_entity)

            expr.append(annotate('company', company_id, 1))

        except (ValueError, TypeError, ObjectDoesNotExist):
            pass

        self.colours = gradient(n=len(expr) + 1)

        queryset = queryset.annotate(
            rank=ExpressionWrapper(
                sum(expr, Value(0)), output_field=IntegerField()
            )
        )

        queryset = (
            queryset.order_by('-rank', 'key')
            .distinct('rank', 'key')
            .values(
                'id',
                'program_id',
                'rank',
                title=F('key'),
                subtitle=F('name'),
                type=Value('default', output_field=CharField()),
            )
        )

        return queryset

    def make_response_data(self, objects):
        for obj in objects:
            if obj['rank']:
                n_matches = bin(obj['rank']).count('1')
                obj['colour'] = self.colours[n_matches]
        return super().make_response_data(objects)


class TaskView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
        DjangoFilterBackend,
    )
    filter_class = TaskFilter

    search_fields = ('key', 'name')

    queryset = models.OracleTask.objects.order_by('key')

    def finalize_queryset(self, qs):

        request = self.request

        expr = []

        try:
            legal_entity = int(request.GET.get('by_legal_entity', None))

            company_id, = models.LegalEntity.objects.values_list(
                'oracle_legal_entity_id'
            ).get(id=legal_entity)

            expr.append(annotate('project__company', company_id, 1))

        except (ValueError, TypeError, ObjectDoesNotExist):
            pass

        try:
            project = int(request.GET.get('by_project', None))
            expr.append(annotate('project_id', project, 2))
        except (ValueError, TypeError):
            pass

        self.colours = gradient(n=len(expr) + 1)

        qs = qs.annotate(
            rank=ExpressionWrapper(
                sum(expr, Value(0)), output_field=IntegerField()
            )
        )

        qs = (
            qs.order_by('-rank', 'key')
            .distinct('rank', 'key')
            .values(
                'id',
                'rank',
                title=F('key'),
                subtitle=F('name'),
                type=Value('default', output_field=CharField()),
            )
        )

        return qs

    def make_response_data(self, objects):
        for obj in objects:
            if obj['rank']:
                n_matches = bin(obj['rank']).count('1')
                obj['colour'] = self.colours[n_matches]
        return super().make_response_data(objects)


class MVPView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('key', 'name')

    queryset = models.OracleMVP.objects.values(
        'id',
        title=F('key'),
        subtitle=F('name'),
        type=Value('default', output_field=CharField()),
    ).order_by('key', 'id')


class ServiceView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name',)

    queryset = models.OracleService.objects.values(
        'id',
        title=F('name'),
        subtitle=F('key'),
        type=Value('default', output_field=CharField()),
    ).order_by('name', 'id')


class ProductLineView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name', 'key')

    queryset = models.OracleProductLine.objects.values(
        'id',
        title=F('name'),
        subtitle=F('key'),
        type=Value('default', output_field=CharField()),
    ).order_by('key', 'name', 'id')


class PurchaseGroupView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name', 'key')

    queryset = models.OraclePurchaseGroup.objects.values(
        'id', 'key', 'name'
    ).order_by('key', 'name', 'id')

    def make_response_data(self, objects):

        output = []

        for obj in objects:

            output.append(
                {
                    'id': obj['id'],
                    'title': '{}: {}'.format(obj['key'], obj['name']),
                    'type': 'default',
                }
            )

        return super().make_response_data(output)


class CompanyView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name', 'shortname', 'inn', 'orgid')

    queryset = models.OracleCompany.objects.values(
        'id',
        title=F('name'),
        subtitle=F('shortname'),
        type=Value('default', output_field=CharField()),
    ).order_by('name', 'id')


class BudgetLineView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name', 'key')

    queryset = models.OracleBudgetLine.objects.values(
        'id',
        title=F('name'),
        subtitle=F('key'),
        type=Value('default', output_field=CharField()),
    ).order_by('name', 'key')


class SystemView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name', 'key')

    queryset = models.OracleSystem.objects.values(
        'id',
        title=F('key'),
        subtitle=F('name'),
        type=Value('default', output_field=CharField()),
    ).order_by('name', 'key')


class SubSystemView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination
    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )
    search_fields = ('name', 'key')
    queryset = models.OracleSubSystem.objects.order_by('key')

    def finalize_queryset(self, qs):

        request = self.request

        expr = []

        try:
            system = int(request.GET.get('by_system', None))
            expr.append(annotate('system_id', system, 1))
        except (ValueError, TypeError):
            pass

        self.colours = gradient(n=len(expr) + 1)

        qs = qs.annotate(
            rank=ExpressionWrapper(
                sum(expr, Value(0)), output_field=IntegerField()
            )
        )

        qs = (
            qs.order_by('-rank', 'key')
            .distinct('rank', 'key')
            .values(
                'id',
                'rank',
                'system',
                title=F('key'),
                subtitle=F('name'),
                type=Value('default', output_field=CharField()),
            )
        )

        return qs

    def make_response_data(self, objects):
        for obj in objects:
            if obj['rank']:
                n_matches = bin(obj['rank']).count('1')
                obj['colour'] = self.colours[n_matches]
        return super().make_response_data(objects)


class CurrencyView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('name', 'key')

    queryset = models.OracleCurrency.objects.values(
        'id',
        title=F('key'),
        subtitle=F('name'),
        type=Value('default', output_field=CharField()),
    ).order_by('key', 'id')


class SupplierView(SuggestMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('full_name', 'inn')

    queryset = models.OracleSupplier.objects.values(
        'id',
        title=F('name'),
        subtitle=F('inn'),
        type=Value('default', output_field=CharField()),
    ).order_by('id')

    def make_response_data(self, objects):

        for obj in objects:
            inn = obj['subtitle'].strip()
            if inn:
                obj['subtitle'] = f'ИНН: {inn}'

        return super().make_response_data(objects)
