# coding: utf-8

import logging
from collections import OrderedDict

from django.conf import settings
from django.utils.translation import get_language
from ids.exceptions import IDSException
from ids.registry import registry
from rest_framework.filters import SearchFilter

from procu.api import models
from procu.api.utils import maybe_fetch_users, strtobool
from procu.rest import generics, pagination
from procu.rest.filters import ExcludeFilter, IncludeFilter, ShowDeletedFilter
from procu.rest.permissions import StaffOnly
from procu.utils.tvm import get_service_ticket, get_user_ticket


logger = logging.getLogger(__name__)


class StaffView(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    # Used for filtering stored users in case of fallback (see Mode II)
    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )
    search_fields = ('first_name', 'last_name', 'email')
    # ----------------------------------------

    def get(self, request, *args, **kwargs):

        lang = get_language()

        # Extract limit and offset
        limit = self.paginator.get_limit(request)
        offset = self.paginator.get_offset(request)
        self.paginator.paginate_queryset([], request)

        # ----------------------------------------------------------------------
        # Mode I:
        # Fetch users specified given by usernames via `id' query parameter.

        usernames = filter(None, request.GET.getlist('id'))

        # Deduplicate values preserving their order
        usernames = list(OrderedDict((x, None) for x in usernames).keys())

        if usernames:

            # Prepare response data
            output = []

            # 1. Trying to find the usernames among those that are already
            #    stored in our database. This way we save an API call for
            #    almost all sensible cases.

            qs = models.User.objects.filter(username__in=usernames)

            # Consider it a success only if all the users are found
            if qs.count() == len(usernames):
                for user in qs:
                    output.append(
                        {
                            'id': user.username,
                            'title': user.full_name,
                            'subtitle': user.username + '@',
                            'type': 'default',
                        }
                    )

            else:
                # 2. Otherwise, fetch users from Staff API.
                #    Intentionally ignore all the nonexistend ones.

                users = maybe_fetch_users(
                    usernames, ['login', 'name', 'phones.number']
                )

                for user in users or []:
                    try:
                        output.append(
                            {
                                'id': user['login'],
                                'title': '{} {}'.format(
                                    user['name']['first'][lang],
                                    user['name']['last'][lang],
                                ),
                                'subtitle': user['login'] + '@',
                                'type': 'default',
                            }
                        )

                    except (KeyError, TypeError):
                        logger.warning(
                            'Unexpected format of user data from Staff API: %s',
                            user,
                        )
                        pass

            return self.paginator.get_paginated_response(output)

        # ----------------------------------------------------------------------
        # Mode II:
        # Fetch a searchable list of users from Intrasearch Suggest.
        # In case of failure fallback to a list of stored staff users.

        # Prepare response data
        output = []

        query = request.GET.get('search', '')

        if strtobool(request.GET.get('as_is')) and bool(query):
            output.append({'id': None, 'title': query, 'type': 'default'})

        repo = registry.get_repository(
            'intrasearch',
            'suggest',
            user_agent='procu',
            service_ticket=get_service_ticket(settings.TVM_ISEARCH_CLIENT_ID),
            user_ticket=get_user_ticket(request),
        )

        # Skip pages that are fully offset:
        # Rewind to the first page contaning items just outside the offset.
        start_page = offset // limit

        # Counters
        remaining = limit
        page = start_page

        # Fallback flag
        has_failed = False

        suggest_kwargs = {
            'version': 1,
            'text': query,
            'layers': 'people',
            'people.per_page': limit,
            'allow_empty': True,
            'language': lang,
        }

        while remaining > 0:
            suggest_kwargs['people.page'] = page

            try:
                chunk = repo.get(suggest_kwargs)[0]['result']

            except IDSException:
                logger.warning('Could not fetch users from Intrasearch Suggest')
                has_failed = True
                break

            chunk = [
                {
                    'id': item['id'],
                    'title': item['title'],
                    'subtitle': item['id'] + '@',
                    'type': 'default',
                }
                for item in chunk
            ]

            if page == start_page:
                # Skip offset that was not covered by skipped pages.
                chunk = chunk[(offset % limit) :]

            if not chunk:
                # User requested more items than the resource has.
                break

            if remaining >= len(chunk):
                output.extend(chunk)
                remaining -= len(chunk)

            else:
                output.extend(chunk[:remaining])
                break

            page += 1

        if has_failed:
            # Fallback to stored users.
            # Apply filters and pagination in a normal DRF way.

            qs = models.User.objects.filter(is_staff=True).order_by(
                'is_deleted', 'last_name', 'id'
            )

            qs = self.filter_queryset(qs)
            qs = self.paginate_queryset(qs)

            for user in qs:
                output.append(
                    {
                        'id': user.username,
                        'title': user.full_name,
                        'subtitle': user.username + '@',
                        'type': 'default',
                    }
                )

        return self.paginator.get_paginated_response(output)
