# coding: utf-8

import logging
import re
from urllib.parse import urlparse

from django.conf import settings
from django.utils.decorators import method_decorator
from django.views.decorators.cache import cache_page
from rest_framework import views
from startrek_client.exceptions import StartrekError

from procu.api.utils import get_tracker_client
from procu.rest import generics, pagination
from procu.rest.permissions import StaffOnly
from .serializers import TicketSerializer

key_regexp = re.compile(r'^/?(\w+)-(\d+)$')

logger = logging.getLogger(__name__)


def maybe_parse_key(search_query):

    # If the search string is a ticket URL, the .path will be '/FOO-XXX'
    # (that is why the following regexp skips the slash).
    # Otherwise the .path will be equal to the search string itself.
    url = urlparse(search_query)
    key_matcher = re.match(key_regexp, url.path)

    if key_matcher:
        # Search by ticket key prefix

        queue, key = key_matcher.groups()
        queue = queue.upper()  # search API is case-sensitive

        return (queue, key)


class TrackerView(views.APIView):
    permission_classes = (StaffOnly,)

    @staticmethod
    def get_by_exact_key(client, search_query):

        parsed_id = maybe_parse_key(search_query)

        if not parsed_id:
            return []

        ticket_id = '-'.join(parsed_id)

        try:
            return list(client.issues.find(keys=[ticket_id]))
        except StartrekError:
            logger.exception('Tracker error')

        return []

    def get_by_queue(self, client, search_query, limit, offset):

        queue_matcher = re.match(r'/?(\w+)-?$', search_query.upper())

        if queue_matcher:
            queue, = queue_matcher.groups()

            try:
                resource = client.issues.find(
                    'Queue: %s' % queue.upper(),
                    per_page=settings.STARTREK_API_PAGE_LIMIT,
                )
            except StartrekError:
                logger.exception('Tracker error')
                resource = []

            return self.paginate_resource(resource, limit=limit, offset=offset)

    def get_by_summary(self, client, search_query, limit, offset):

        try:
            resource = client.issues.find(
                query='Summary: "%s"' % search_query,
                per_page=settings.STARTREK_API_PAGE_LIMIT,
            )
        except StartrekError:
            logger.exception('Tracker error')
            resource = []

        return self.paginate_resource(resource, limit=limit, offset=offset)

    def get_by_keys(self, client, limit, offset):

        try:
            resource = client.issues.find(
                per_page=settings.STARTREK_API_PAGE_LIMIT,
                **self.get_find_kwargs(),
            )

        except StartrekError:
            logger.exception('Tracker error')
            resource = []

        return self.paginate_resource(resource, limit=limit, offset=offset)

    def get_suggest(self, limit, offset):

        search_query = self.request.GET.get('search', '')

        client = get_tracker_client(request=self.request)

        tickets = self.get_by_queue(client, search_query, limit, offset)

        if tickets:
            return tickets

        tickets = self.get_by_exact_key(client, search_query)
        limit -= len(tickets)

        tickets.extend(self.get_by_keys(client, limit, offset))

        if tickets:
            return tickets

        return self.get_by_summary(client, search_query, limit, offset)

    def get(self, request):

        # Extract limit and offset query parameters the same way like elsewhere.
        paginator = pagination.LimitOffsetPagination()
        limit = paginator.limit = paginator.get_limit(request)
        offset = paginator.offset = paginator.get_offset(request)

        # Dummy value, not used
        paginator.count = 0

        tickets = self.get_suggest(limit, offset)

        return paginator.get_paginated_response(
            TicketSerializer(tickets, many=True).data
        )

    @staticmethod
    def paginate_resource(resource, offset, limit):

        st_limit = settings.STARTREK_API_PAGE_LIMIT

        if isinstance(resource, list):
            # Result fitting on a single page is returned as an ordinary list.
            return resource[offset : offset + limit]

        # Skip pages that are fully offset:
        # Rewind to the first page contaning items just outside the offset.
        start_page = offset // st_limit + 1

        # Counters
        remaining = limit
        page = start_page

        result = []

        while remaining > 0:
            chunk = list(resource.get_page(page))

            if page == start_page:
                # Skip offset that was not covered by skipped pages.
                chunk = chunk[(offset % st_limit) :]

            if not chunk:
                # User requested more items than the resource has.
                break

            if remaining >= len(chunk):
                result.extend(chunk)
                remaining -= len(chunk)

            else:
                result.extend(chunk[:remaining])
                break

            page += 1

        return result

    def get_find_kwargs(self):
        # Use `id' query parameter for picking ticket by keys.
        keys = [
            key.upper()
            for key in self.request.GET.getlist('id')
            if re.match(key_regexp, key)
        ]
        if keys:
            return {'keys': keys}

        query = self.get_ticket_query(self.request.GET.get('search', ''))

        return {'query': '(%s)' % query}

    @staticmethod
    def get_ticket_query(search_query):

        parsed_id = maybe_parse_key(search_query)

        if parsed_id:
            # Search by ticket key prefix
            queue, key = parsed_id

            lower = upper = '%s-%s' % (queue, key)
            length = len(key)

            ranges = [(lower, upper)]

            # The Tracker does not have a built-in search by key prefixes,
            # so we emulate it by a compound query.
            #
            # Example: "FOO-XXX" results in a query searching within "FOO"
            # for keys in ranges [XXX], [XXX0, XXX9], [XXX00, XXX99], and so on.

            while length < 5:
                lower += '0'
                upper += '9'
                ranges.append((lower, upper))
                length += 1

            ticket_query = " or ".join(
                map(lambda x: '(Key: >= "%s" and Key: <= "%s")' % x, ranges)
            )

            ticket_query = '(%s) and Key: !%s-%s' % (ticket_query, queue, key)

        else:
            # Fallback to text search within ticket titles.
            ticket_query = 'Summary: "%s"' % search_query

        return ticket_query


class TrackerComponentView(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.NoPagination

    def get(self, request):

        try:
            queue = request.GET.get('queue', '')

            if not queue:
                raise StartrekError('No queue')

            client = get_tracker_client()
            components = [c['name'] for c in client.queues[queue]['components']]

            ids = set(self.request.GET.getlist('id'))

            data = [
                {'id': name, 'title': name, 'type': 'default'}
                for name in components
                if not ids or (name in ids)
            ]

        except StartrekError:
            logger.exception('Tracker error')
            data = []

        return self.get_paginated_response(data)


class QueueView(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    @method_decorator(cache_page(60 * 60))
    def get(self, request):

        client = get_tracker_client()
        queues = [
            {'id': q.key.upper(), 'title': q.key.upper()} for q in client.queues
        ]

        # -----------------------------------------------------------------

        ids = [key.upper() for key in request.GET.getlist('id')]
        if ids:
            queues = [q for q in queues if q['id'] in ids]

        # -----------------------------------------------------------------

        query = request.GET.get('search', None)

        if query is not None:
            query = query.upper()
            queues = [q for q in queues if query in q['id']]

        # -----------------------------------------------------------------

        queues = self.paginate_queryset(queues)

        return self.get_paginated_response(queues)


class IssueTypeView(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    @method_decorator(cache_page(60 * 60))
    def get(self, request):

        try:
            queue = request.GET['queue'].strip()

            if not queue:
                raise ValueError

            client = get_tracker_client()
            queue = client.queues[queue]

            issuetypes = [
                {'id': issuetype.key, 'title': issuetype.name}
                for issuetype in queue.issuetypes
            ]

            # Put default type to the top of the list
            issuetypes.sort(key=lambda x: x['id'] != queue.defaultType.key)

        except (
            KeyError,  # No queue supplied
            ValueError,  # Invalid input
            AttributeError,  # Unexpected queue object
            StartrekError,  # Access denied, invalid queue, etc.
        ):
            issuetypes = []

        # ----------------------------------------------------------------

        ids = [key for key in request.GET.getlist('id')]
        if ids:
            issuetypes = [q for q in issuetypes if q['id'] in ids]

        # -----------------------------------------------------------------

        query = request.GET.get('search', None)

        if query is not None:
            query = query.lower()
            issuetypes = [q for q in issuetypes if query in q['title'].lower()]

        # -----------------------------------------------------------------

        issuetypes = self.paginate_queryset(issuetypes)

        return self.get_paginated_response(issuetypes)
