# coding: utf-8

import logging

from rest_framework.filters import SearchFilter

from procu.api import models
from procu.rest import generics, pagination
from procu.rest.filters import ExcludeFilter, IncludeFilter, ShowDeletedFilter
from procu.rest.permissions import StaffOnly
from ..mixins import SuggestMixin

logger = logging.getLogger(__name__)


class UserSuggectMixin(SuggestMixin):
    def make_response_data(self, objects):

        output = []

        for obj in objects:

            full_name = (obj.first_name + ' ' + obj.last_name).strip()
            username = obj.username + ('@' if obj.is_staff else '')

            if not full_name:
                title = username
                subtitle = ''

            else:
                title = full_name
                subtitle = username

            output.append(
                {
                    'id': obj.id,
                    'title': title,
                    'subtitle': subtitle,
                    'is_deleted': obj.is_deleted,
                    'type': 'default',
                }
            )

        return super().make_response_data(output)


# ------------------------------------------------------------------------------


class StaffUsersView(UserSuggectMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('first_name', 'last_name', 'email')

    def get_queryset(self):

        # Only users that can make progress in procurements
        valid_usernames = (
            models.UserGroup.objects.values_list('username', flat=True)
            .filter(group__permissions__codename='progress_enquiry')
            .distinct('username')
        )

        return (
            models.User.objects.filter(
                is_staff=True, username__in=valid_usernames
            )
            .only(
                'id',
                'username',
                'first_name',
                'last_name',
                'email',
                'is_staff',
                'is_deleted',
            )
            .order_by('is_deleted', 'last_name', 'id')
        )


class ContactsView(UserSuggectMixin, generics.ListAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('first_name', 'last_name', 'email', 'supplier__title')

    queryset = (
        models.User.objects.filter(is_staff=False)
        .only(
            'id',
            'username',
            'first_name',
            'last_name',
            'email',
            'is_staff',
            'is_deleted',
        )
        .order_by('is_deleted', 'last_name', 'id')
    )


class AllUsersView(UserSuggectMixin, generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    pagination_class = pagination.LimitOffsetPagination

    filter_backends = (
        SearchFilter,
        IncludeFilter,
        ExcludeFilter,
        ShowDeletedFilter,
    )

    search_fields = ('first_name', 'last_name', 'email')

    def get_queryset(self):
        return models.User.objects.only(
            'id',
            'username',
            'first_name',
            'last_name',
            'email',
            'is_staff',
            'is_deleted',
        ).order_by('is_deleted', 'last_name', 'id')
