# coding: utf-8

from django.utils.translation import gettext_lazy as _
from rest_framework import validators

from procu.api import models
from procu.api.common.fields import PaymentTermsField
from procu.api.enums import RISK
from procu.api.spark import tasks
from procu.api.user.serializers import UserBrief
from procu.rest import serializers


# ------------------------------------------------------------------------------


class SupplierBrief(serializers.ModelSerializer):
    class Meta:
        model = models.Supplier
        fields = ('id', 'title', 'legal_name', 'is_cold', 'is_deleted')


# ------------------------------------------------------------------------------


class Info(serializers.Serializer):

    altname = serializers.CharField(
        max_length=255,
        label=_('SUPPLIER_INFO::ALTNAME'),
        required=False,
        allow_blank=False,
    )
    kpp = serializers.CharField(
        max_length=255,
        label=_('SUPPLIER_INFO::KPP'),
        required=False,
        allow_blank=False,
    )
    legal_address = serializers.CharField(
        label=_('SUPPLIER_INFO::LEGAL_ADDRESS'),
        required=False,
        allow_blank=False,
    )
    actual_address = serializers.CharField(
        label=_('SUPPLIER_INFO::ACTUAL_ADDRESS'),
        required=False,
        allow_blank=False,
    )
    general_director = serializers.CharField(
        max_length=255,
        label=_('SUPPLIER_INFO::GENERAL_DIRECTOR'),
        required=False,
        allow_blank=False,
    )
    accountant = serializers.CharField(
        max_length=255,
        label=_('SUPPLIER_INFO::ACCOUNTANT'),
        required=False,
        allow_blank=False,
    )
    url = serializers.CharField(
        max_length=1023,
        label=_('SUPPLIER_INFO::URL'),
        required=False,
        allow_blank=False,
    )


# ------------------------------------------------------------------------------


class CreateCold(serializers.Serializer):
    email = serializers.EmailField(
        validators=[
            validators.UniqueValidator(
                queryset=models.User.objects.filter(is_deleted=False),
                message=_('SUPPLIER_COLD::ERROR_CONTACT_ALREADY_EXISTS'),
                lookup='iexact',
            )
        ],
        max_length=255,
        label=_('SUPPLIER_COLD::EMAIL'),
    )
    title = serializers.CharField(
        validators=[
            validators.UniqueValidator(
                queryset=models.Supplier.objects.filter(is_deleted=False),
                message=_('SUPPLIER_COLD::ERROR_SUPPLIER_ALREADY_EXISTS'),
                lookup='iexact',
            )
        ],
        max_length=255,
        label=_('SUPPLIER_COLD::TITLE'),
    )


# ------------------------------------------------------------------------------

# Avoid cyclic imports
from procu.api.supplier_tag.serializers import SupplierTagBriefSerializer
from .fields import AgentField, RiskField
from .validators import validate_vat_id
from .warning.serializers import List as WarningList


class OracleSupplierBrief(serializers.ModelSerializer):
    class Meta:
        model = models.OracleSupplier
        fields = ('id', 'key', 'name', 'inn')


class SupplierBase(serializers.ModelSerializer):
    check_for_nested_writes = False

    agents = AgentField(
        many=True, required=False, label=_('SUPPLIER::CONTACTS')
    )
    vat_id = serializers.CharField(
        validators=[validate_vat_id], required=True, allow_blank=True
    )
    tags = serializers.PKPrettyField(
        queryset=models.SupplierTag.objects,
        serializer=SupplierTagBriefSerializer,
        many=True,
        allow_empty=True,
        required=False,
        label=_('SUPPLIER::TAGS'),
    )
    info = Info(required=False)
    risk = RiskField(label=_('MODEL_SUPPLIER::RISK'))
    author = UserBrief(read_only=True)
    terms = PaymentTermsField()
    oracle_supplier = serializers.PKPrettyField(
        serializer=OracleSupplierBrief,
        queryset=models.OracleSupplier.objects.all(),
        allow_null=True,
    )

    warnings = WarningList(
        many=True, read_only=True, label=_('SUPPLIER::WARNINGS')
    )

    def save(self, **kwargs):
        kwargs = dict(kwargs)
        data = self.validated_data

        update_risks = False

        if data.get('vat_id'):
            if self.instance:
                update_risks = self.instance.vat_id != data['vat_id']
            else:
                update_risks = True

            if update_risks:
                kwargs['risk'] = RISK.FETCHING
                kwargs['risk_data'] = {}

        else:
            kwargs['risk'] = RISK.UNKNOWN
            kwargs['risk_data'] = {}
            kwargs['risk_updated_at'] = None

        instance = super().save(**kwargs)

        if update_risks:
            tasks.update_supplier_risks.apply_async(
                kwargs={'supplier_id': instance.id, 'update': False}
            )

        return instance

    class Meta:
        model = models.Supplier
        fields = (
            'id',
            'title',
            'legal_name',
            'has_contract',
            'terms',
            'vat_id',
            'comment',
            'agents',
            'tags',
            'info',
            'secrets',
            'risk',
            'is_cold',
            'author',
            'oracle_supplier',
            'can_pay_by_card',
            'warnings',
            'is_deleted',
        )
        read_only_fields = ('is_cold',)
        extra_kwargs = {'is_deleted': {'label': _('SUPPLIER::IS_DELETED')}}


class SupplierSecrets(SupplierBase):
    pass


class SupplierNoSecrets(SupplierBase):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields.pop('secrets', None)


# ------------------------------------------------------------------------------


class List(serializers.ModelSerializer):

    agents = serializers.SerializerMethodField(label=_('SUPPLIER::CONTACTS'))
    tags = serializers.SerializerMethodField(label=_('SUPPLIER::TAGS'))

    has_warnings = serializers.BooleanField(read_only=True)
    risk = RiskField()

    def get_agents(self, instance):
        return self.context['agents'][instance.id]

    def get_tags(self, instance):
        return self.context['tags'][instance.id]

    class Meta:
        model = models.Supplier
        fields = (
            'id',
            'title',
            'has_contract',
            'vat_id',
            'agents',
            'tags',
            'is_cold',
            'can_pay_by_card',
            'has_warnings',
            'risk',
            'is_deleted',
        )


# ------------------------------------------------------------------------------


class SupplierRisks(serializers.ModelSerializer):
    risk = RiskField(label=_('MODEL_SUPPLIER::RISK'))

    class Meta:
        model = models.Supplier
        fields = ('id', 'risk', 'risk_data', 'risk_updated_at')
