# coding: utf-8

from datetime import date, timedelta
from itertools import groupby
from defusedxml import ElementTree

from celery import shared_task
from celery.utils.log import get_task_logger
from django.conf import settings
from django.core import mail
from django.db.models import Max, Q
from django.db.transaction import atomic
from django.utils import dateparse, timezone, translation
from django.utils.encoding import force_text
from startrek_client.exceptions import StartrekError

from procu.api import models
from procu.api.enums import ER, ES, QS
from procu.api.push.internal import notify_enquiry_expired
from procu.api.saving.utils import update_savings
from procu.api.utils import dict_diff, get_tracker_client, json_dumps
from procu.api.utils.decimal import money
from procu.utils.lock import locked
from procu.wf import tasks


@shared_task
@locked(name='update_exchange_rates')
def update_exchange_rates():
    import requests

    logger = get_task_logger(__name__)

    today = timezone.now().strftime('%d/%m/%Y')
    url = settings.CURRENCY_RATES_URL.format(today=today)

    try:
        source = requests.get(url).text

    except requests.RequestException:
        logger.exception('Failed to fetch currency data')
        raise

    root = ElementTree.fromstring(source)
    rates = {}

    try:
        for elem in root.iter('Valute'):
            currency = dict(map(lambda e: (e.tag, e.text), elem))
            currency['Value'] = money(currency['Value'].replace(',', '.'))
            currency['Nominal'] = int(currency['Nominal'])
            rates[currency['CharCode']] = currency

    except (KeyError, ValueError):
        logger.error('Failed to parse currency data')
        raise

    updated = []

    for instance in models.Currency.objects.filter(is_deleted=False):

        if instance.char_code == 'RUB':
            continue

        try:
            rate = rates[instance.char_code]
            instance.nominal = rate['Nominal']
            instance.rate = rate['Value']
            instance.save()

            updated.append(instance.char_code)

        except KeyError:
            logger.warning(
                "Currency `%s' is unknown to CBR" % instance.char_code
            )

    logger.info(
        'Successfully updated currencies: {codes}'.format(
            codes=', '.join(updated)
        )
    )

    return updated


@shared_task
@locked(name='switch_to_review')
def switch_to_review():

    logger = get_task_logger(__name__)
    translation.activate('ru')

    user = models.User.objects.get_or_create_by_login(settings.ROBOT_LOGIN)

    with atomic():

        current = timezone.now()

        qs = models.Quote.objects.all()

        candidates = list(
            qs.filter(
                Q(deadline_at__lt=current, status=QS.BIDDING)
                | Q(deadline_at__gt=current, status=QS.REVIEW)
            )
            .filter(request__isnull=False)
            .values_list('request__enquiry_id', flat=True)
        )

        olds = list(
            models.Enquiry.objects.values('id', 'status', 'reason').filter(
                id__in=candidates
            )
        )

        qs.filter(deadline_at__lt=current, status=QS.BIDDING).update(
            status=QS.REVIEW
        )
        qs.filter(deadline_at__gt=current, status=QS.REVIEW).update(
            status=QS.BIDDING
        )

        news = list(
            models.Enquiry.objects.values('id', 'status', 'reason').filter(
                id__in=candidates
            )
        )

    for old, new in zip(olds, news):

        old_context = {
            'status': force_text(ES.i18n[old['status']]),
            'reason': force_text(ER.i18n[old['reason']]),
        }

        new_context = {
            'status': force_text(ES.i18n[new['status']]),
            'reason': force_text(ER.i18n[new['reason']]),
        }

        diff = dict_diff(old_context, new_context, show_unchanged=False)

        if diff:
            models.Log.objects.create(
                enquiry_id=old['id'],
                user=user,
                type='update_enquiry',
                data=json_dumps(diff),
            )

            logger.info('Enquiry %d has been updated', old['id'])

            if (old['status'], new['status']) == (ES.BIDDING, ES.REVIEW):
                messages = notify_enquiry_expired(old['id'])
                with mail.get_connection(fail_silently=True) as conn:
                    conn.send_messages(messages)


@shared_task
@locked(name='fix_formatting')
def fix_formatting():
    tasks.fix_formatting()


@shared_task
@locked(name='sync_remotes')
def sync_remotes(enquiry_id, link_to, unlink_from, request=None):
    logger = get_task_logger(__name__)
    client = get_tracker_client(request=request)

    enquiry_key = 'YP{}'.format(enquiry_id)

    # --------------------------------------------------------------------------
    # Unlink enquiry

    for ticket in unlink_from:

        try:
            issue = client.issues[ticket]
            links = issue.remotelinks.get_all(
                origin=settings.STARTREK_LINK_ORIGIN
            )

            for remote in links:
                if remote.object.key == enquiry_key:
                    remote.delete()
                    break

        except StartrekError:
            logger.warning('Could not unlink %s from %s', enquiry_key, ticket)
            continue

    # --------------------------------------------------------------------------
    # Link enquiry

    for ticket in link_to:

        try:
            issue = client.issues[ticket]

            issue.remotelinks.create(
                origin=settings.STARTREK_LINK_ORIGIN,
                key=enquiry_key,
                relationship='relates',
                params={'backlink': False},
            )

        except StartrekError:
            logger.warning('Could not link %s to %s', enquiry_key, ticket)
            continue


@shared_task
@locked(name='update_savings_task')
def update_savings_task(*args, **kwargs):
    update_savings(*args, **kwargs)


@shared_task
@locked(name='make_enquiry_log')
def make_enquiry_log():

    qs = (
        models.EnquiryLog.objects.values(
            'enquiry_id', 'status', 'reason', 'state'
        )
        .distinct('enquiry_id')
        .order_by('enquiry_id', '-id')
    )
    logs = {x.pop('enquiry_id'): x for x in qs}

    qs = models.Enquiry.objects.values('id', 'status', 'reason', 'state')

    for new in qs:

        enquiry_id = new.pop('id')
        old = logs.get(enquiry_id)

        if old != new:
            models.EnquiryLog.objects.create(enquiry_id=enquiry_id, **new)


@shared_task
@locked(name='update_filter_idx')
def update_filter_idx(include_closed=False):
    logger = get_task_logger(__name__)

    if include_closed:
        status_condition = Q()
    else:
        status_condition = ~Q(request__status=ES.CLOSED)

    all_quotes = (
        models.Quote.objects.values(
            'request__enquiry_id', 'supplier_id', 'has_won', 'has_offer'
        ).filter(
            status_condition,
            request__enquiry_id__isnull=False,
            is_deleted=False,
        )
        # Ordering matters for the following `groupby`!
        .order_by('request__enquiry_id')
    )

    data = {
        enquiry: sorted(group, key=lambda x: x['supplier_id'])
        for enquiry, group in groupby(
            all_quotes, key=lambda x: x.pop('request__enquiry_id')
        )
    }

    with atomic():
        enquiries = {
            enquiry.id: enquiry
            for enquiry in models.Enquiry.objects.only('id', 'filter_idx')
        }

        for id_, quotes in data.items():

            idx = {
                'suppliers': [q['supplier_id'] for q in quotes],
                'has_won': [q['supplier_id'] for q in quotes if q['has_won']],
                'has_offer': [
                    q['supplier_id'] for q in quotes if q['has_offer']
                ],
            }

            enquiry = enquiries[id_]

            if enquiry.filter_idx != idx:
                enquiry.filter_idx = idx
                enquiry.save(update_fields=('filter_idx',))

    logger.info('Filter indexes have been updated')


@shared_task
@locked(name='fetch_buy')
def fetch_buy():
    logger = get_task_logger(__name__)

    m = models.BUYTicket.objects.aggregate(latest=Max('created_at'))
    start = m['latest']

    if start is None:
        start = date(2017, 5, 1)
    else:
        start = timezone.localtime(start).date() + timedelta(days=1)

    today = date.today()

    client = get_tracker_client()

    issues = list(
        client.issues.find(
            f'Queue: {settings.BUY_QUEUE}, {settings.DCBUY_QUEUE} '
            f'AND Created: >= {start.isoformat()} '
            f'AND Created: < {today.isoformat()} '
            f'"Sort By": key ASC',
            per_page=200,
        )
    )

    objs = models.BUYTicket.objects.bulk_create(
        models.BUYTicket(
            key=issue.key, created_at=dateparse.parse_datetime(issue.createdAt)
        )
        for issue in issues
    )

    logger.info(f'{len(objs)} new tickets fetched')
