# coding: utf-8

import logging
import re

from django.conf import settings
from django.db.models import F
from rest_framework import response, status
from rest_framework.filters import OrderingFilter
from startrek_client.exceptions import StartrekError

from procu.api import models, tasks
from procu.api.enquiry.filters import YPSearchFilter
from procu.api.enums import LINK
from procu.api.utils import get_tracker_client, json_dumps
from procu.rest import exceptions, generics, pagination
from procu.rest.authentication import MixedAuthenticationIgnoreScopes
from procu.rest.permissions import StaffOnly
from . import serializers
from .filters import ID_REGEX, YPFilter

logger = logging.getLogger(__name__)


class EnquiryView(generics.ListAPIView):
    permission_classes = (StaffOnly,)
    serializer_class = serializers.EnquirySerializer
    pagination_class = pagination.LimitOffsetPagination
    authentication_classes = (MixedAuthenticationIgnoreScopes,)

    def get_queryset(self):
        return (
            models.Enquiry.objects.permitted(self.request.user)
            .annotate(deadline_at=F('request__deadline_at'))
            .select_related('manager')
        )

    filter_backends = (YPSearchFilter, OrderingFilter, YPFilter)
    ordering = ('-updated_at',)
    search_fields = ('id', 'subject')


class EnquiryLinkCreate(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    serializer_class = serializers.Link
    authentication_classes = (MixedAuthenticationIgnoreScopes,)

    def post(self, request, *args, **kwargs):

        try:
            enquiry_id = ID_REGEX.match(kwargs['enquiry_slug']).group(2)

        except AttributeError:
            raise exceptions.ValidationError(
                '%s is not an enquiry id' % kwargs['enquiry_slug']
            )

        enquiry = generics.get_object_or_404(
            models.Enquiry.objects.permitted(self.request.user).all(),
            pk=enquiry_id,
        )

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        ticket = serializer.validated_data['ticket']

        models.Link.objects.get_or_create(
            enquiry=enquiry,
            key=ticket,
            defaults={'type': LINK.TRACKER, 'author': request.user},
        )

        models.Log.objects.create(
            quote=None,
            enquiry_id=enquiry.id,
            user=self.request.user,
            type='create_enquiry_links',
            data=json_dumps({'keys': [ticket]}),
            new=json_dumps({'keys': [ticket]}),
        )

        return response.Response([ticket], status=status.HTTP_201_CREATED)


class EnquiryLinkDelete(generics.GenericAPIView):
    permission_classes = (StaffOnly,)
    authentication_classes = (MixedAuthenticationIgnoreScopes,)

    def delete(self, request, *args, **kwargs):

        try:
            enquiry_id = ID_REGEX.match(kwargs['enquiry_slug']).group(2)

        except AttributeError:
            raise exceptions.ValidationError(
                '%s is not an enquiry id' % kwargs['enquiry_slug']
            )

        enquiry = generics.get_object_or_404(
            models.Enquiry.objects.permitted(self.request.user).all(),
            pk=enquiry_id,
        )

        ticket = kwargs['ticket']

        cnt, _ = models.Link.objects.filter(
            enquiry=enquiry, key=ticket, type=LINK.TRACKER
        ).delete()

        if cnt:
            models.Log.objects.create(
                quote=None,
                enquiry_id=enquiry.id,
                user=self.request.user,
                type='remove_enquiry_links',
                data=json_dumps({'keys': [ticket]}),
                new=json_dumps({'keys': [ticket]}),
            )

        return response.Response(status=status.HTTP_204_NO_CONTENT)


class SkipEvent(Exception):
    pass


def get_delivery_order(text):
    m = re.search(r'(1\d{7})', text)
    return m.group(1) if m else None


def maybe_process_comment(quote, changelog):

    delivery_orders = set(quote.delivery_orders)

    if not changelog.comments:
        return

    for comment in changelog.comments.get('added', ()):
        number = get_delivery_order(comment.text)
        if number:
            delivery_orders.add(number)

    for comment in changelog.comments.get('updated', ()):
        old = get_delivery_order(comment['from'])
        new = get_delivery_order(comment['to'])

        if old:
            delivery_orders.discard(old)

        if new:
            delivery_orders.add(new)

    quote.delivery_orders = sorted(delivery_orders)
    quote.save(update_fields=('delivery_orders',))


def maybe_process_link(quote, changelog):

    if not changelog.links:
        return

    link_to = []
    unlink_from = []

    for link in changelog.links:

        if link['to'] and not link['from']:
            link_to.append(link['to']['object'].key)

        if link['from'] and not link['to']:
            unlink_from.append(link['from']['object'].key)

    link_to = [k for k in link_to if k.startswith(settings.ACC_QUEUE)]
    unlink_from = [k for k in unlink_from if k.startswith(settings.ACC_QUEUE)]

    enquiry_id = models.Request.objects.values_list(
        'enquiry_id', flat=True
    ).get(quotes=quote.id)

    tasks.sync_remotes.apply_async(
        kwargs={
            'enquiry_id': enquiry_id,
            'link_to': link_to,
            'unlink_from': unlink_from,
        }
    )

    # --------------------------------------------------------------------------

    user = models.User.objects.get_or_create_by_login(settings.ROBOT_LOGIN)

    # Unlink tickets from the quote
    models.SCSTicket.objects.filter(
        quote=quote, ticket__in=unlink_from
    ).delete()

    qs = models.Link.objects.filter(
        enquiry_id=enquiry_id, type=LINK.TRACKER, key__in=unlink_from
    )
    unlinked = list(qs.values_list('key', flat=True))

    qs.delete()

    if unlinked:
        models.Log.objects.create(
            enquiry_id=enquiry_id,
            user=user,
            type='remove_enquiry_links',
            data=json_dumps({'keys': unlinked}),
            new=json_dumps({'keys': unlinked}),
        )

    # -----------------------------------------------------

    # Link tickets to the quote
    models.SCSTicket.objects.bulk_create(
        [models.SCSTicket(quote=quote, ticket=ticket) for ticket in link_to]
    )

    linked = []

    for key in link_to:
        _, created = models.Link.objects.get_or_create(
            enquiry_id=enquiry_id,
            key=key,
            defaults={'type': LINK.TRACKER, 'author': user},
        )

        if created:
            linked.append(key)

    if linked:
        models.Log.objects.create(
            enquiry_id=enquiry_id,
            user=user,
            type='create_enquiry_links',
            data=json_dumps({'keys': linked}),
            new=json_dumps({'keys': linked}),
        )


class SCSEventView(generics.GenericAPIView):
    permission_classes = ()
    serializer_class = serializers.HookSerializer
    authentication_classes = ()

    def post(self, request, *args, **kwargs):

        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        data = serializer.validated_data

        try:
            changelog_id = data['event']['id']
            issue_key = data['issue']['key']

            scs_ticket = models.SCSTicket.objects.get(ticket=issue_key)
            quote = scs_ticket.quote

            client = get_tracker_client()

            try:
                changelog = client.issues[issue_key].changelog[changelog_id]

                maybe_process_comment(quote, changelog)
                maybe_process_link(quote, changelog)

            except (StartrekError, IndexError):
                logger.exception('Could not handle webhook')
                return response.Response({'results': 'error'})

            # ------------------------------------------------------------------

            return response.Response({'results': 'ok'})

        except models.SCSTicket.DoesNotExist:
            return response.Response({'results': 'notfound'})
