# coding: utf-8

from django.utils.translation import gettext_lazy as _
from rest_framework.exceptions import ValidationError

from procu.api import models
from procu.rest import serializers


class UserCurrent(serializers.ModelSerializer):
    full_name = serializers.CharField()

    class Meta:
        model = models.User
        fields = ('id', 'username', 'full_name', 'is_staff')


class UserBrief(serializers.ModelSerializer):
    full_name = serializers.CharField()
    is_clickable = serializers.SerializerMethodField()

    class Meta:
        model = models.User
        fields = (
            'id',
            'username',
            'email',
            'full_name',
            'first_name',
            'last_name',
            'is_staff',
            'is_deleted',
            'is_clickable',
        )

    def get_is_clickable(self, instance):
        try:
            user = self.context['request'].user
            return user.is_staff and user.has_perm('api.retrieve_user')

        except (KeyError, TypeError, AttributeError):
            return False


class UserChangePasswordSerializer(serializers.Serializer):

    old_password = serializers.CharField(
        allow_blank=False,
        max_length=255,
        required=True,
        label=_('CHANGE_PASSWORD::OLD_PASSWORD'),
    )
    new_password = serializers.CharField(
        allow_blank=False,
        max_length=255,
        required=True,
        label=_('CHANGE_PASSWORD::NEW_PASSWORD'),
    )
    new_password_confirmation = serializers.CharField(
        allow_blank=False,
        max_length=255,
        required=True,
        label=_('CHANGE_PASSWORD::NEW_PASSWORD_CONFIRMATION'),
    )

    def validate(self, data):

        user = self.context['request'].user

        if not user.check_password(data['old_password']):
            raise ValidationError(
                {'old_password': _('CHANGE_PASSWORD::INCORRECT_PASSWORD')}
            )

        if data['new_password'] != data['new_password_confirmation']:
            raise ValidationError(
                {'new_password_confirmation': _('CHANGE_PASSWORD::NO_MATCH')}
            )

        return data


class LoginSerializer(serializers.Serializer):

    username = serializers.EmailField(
        allow_blank=False,
        max_length=255,
        required=True,
        label=_('AUTH_EXTERNAL::EMAIL'),
    )
    password = serializers.CharField(
        allow_blank=False,
        max_length=255,
        required=True,
        label=_('AUTH_EXTERNAL::PASSWORD'),
    )


class QuickLoginSerializer(serializers.Serializer):

    email = serializers.EmailField(
        allow_blank=False,
        max_length=255,
        required=False,
        label=_('AUTH_EXTERNAL::EMAIL'),
    )
