# coding: utf-8

import decimal
import logging
import secrets
from collections import OrderedDict
from datetime import datetime, timedelta
from functools import partial
from typing import List, Optional, Sequence, Tuple

from django.conf import settings
from django.core.cache import cache
from django.db.models import Case, IntegerField, Value, When
from ids.exceptions import IDSException
from ids.registry import registry
from rest_framework.renderers import JSONRenderer

from procu.utils.tvm import get_service_ticket

decimal.setcontext(decimal.Context(traps=[decimal.InvalidOperation]))

logger = logging.getLogger(__name__)


def is_previewable(filename):
    pos = filename.rfind('.')
    if pos > -1:
        extension = filename[pos + 1 :]
        return extension.lower() in settings.PREVIEWABLE_FILE_TYPES
    return False


def annotate(field, value, mask):
    return Case(
        When(then=Value(mask), **{field: value}),
        default=Value(0),
        output_field=IntegerField(),
    )


def annotate_rank(qs, field, value):

    rank_field = f'{field}_rank'

    qs = qs.annotate(
        **{
            rank_field: Case(
                When(then=Value(0), **{field: value}),
                default=Value(1),
                output_field=IntegerField(),
            )
        }
    )

    ordering = tuple(dict.fromkeys((rank_field, *qs.query.order_by)))

    return qs.order_by(*ordering)


def get_front_version():
    return cache.get_or_set(
        'FRONT_RANDOM', partial(secrets.token_hex, 8), timeout=None
    )


def maybe_fetch_users(usernames: Sequence, fields: Sequence) -> Optional[List]:

    try:
        repo = registry.get_repository(
            'staff',
            'person',
            user_agent='procu',
            service_ticket=get_service_ticket(settings.TVM_STAFF_API_CLIENT_ID),
        )

        return repo.get(
            {'login': ','.join(usernames), '_fields': ','.join(fields)}
        )

    except (IDSException):
        pass


def dict_diff(old, new, fields=None, show_unchanged=True):

    universe = set(old) | set(new)

    if fields is None:
        fields = universe

    diff = OrderedDict()

    # Dictionaries match
    if all(old[k] == new[k] for k in set(fields) & set(old) & set(new)):
        return diff

    for key in fields:
        if key not in universe:
            continue

        try:
            if old[key] == new[key]:
                if show_unchanged:
                    diff[key] = old[key]
                continue
        except KeyError:
            pass

        diff[key] = {}

        if key in old:
            diff[key]['__old__'] = old[key]

        if key in new:
            diff[key]['__new__'] = new[key]

    return diff


def json_dumps(obj):
    renderer = JSONRenderer()
    return renderer.render(obj).decode()


def is_readonly():
    return cache.get(settings.SWITCH_READONLY, False)


def get_maintenance() -> Optional[Tuple[datetime, timedelta]]:
    try:
        started_at, downtime = cache.get(settings.SWITCH_MAINTENANCE)

        if type(started_at) is not datetime or type(downtime) is not timedelta:
            raise ValueError

        return (started_at, downtime)

    except (TypeError, ValueError):
        return None


def strtobool(val, default=False):
    val = str(val).lower().strip()

    if val in ('y', 'yes', 't', 'true', 'on', '1'):
        return True
    elif val in ('n', 'no', 'f', 'false', 'off', '0'):
        return False

    return default
